# -*- encoding: utf-8 -*-
import logging
import tornado.escape
import ujson as json
import zlib
from tornado.httpclient import HTTPResponse
from tornado.httputil import HTTPHeaders

from travel.avia.api_gateway.application.fetcher import Fetcher
from travel.avia.api_gateway.application.fetcher.anywhere_landing.memcache import AnywhereLandingCache
from travel.avia.api_gateway.settings import TRAVEL_API_URL
from travel.avia.library.python.urls.search import C


logger = logging.getLogger(__name__)


class CountryRestrictionsDataFetcher(Fetcher):
    def __init__(self, request_headers, from_id, to_ids, *args, **kwargs):
        # type: (CountryRestrictionsDataFetcher, HTTPHeaders, int, list[int]) -> None
        super(CountryRestrictionsDataFetcher, self).__init__(*args, **kwargs)
        self._from_id = from_id
        self._to_ids = to_ids

    def fetch(self, fetchers=None):  # type: (List[Fetcher]) -> None
        headers = {
            'Content-Type': 'application/json',
        }

        cached_data = AnywhereLandingCache.get_country_restrictions(self._from_id)
        if cached_data:
            result = json.loads(zlib.decompress(cached_data))
            self.finish_callback(result, field=self.field)
        else:
            self.request(
                f'{TRAVEL_API_URL}/avia_country_restrictions/v1/data-by-list/',
                body=tornado.escape.json_encode({
                    'from_point_key': str(C(self._from_id)),
                    'to_point_keys': [str(C(city_id)) for city_id in self._to_ids],
                }),
                headers=headers,
                method='POST',
                callback=self.on_response,
            )

    def on_response(self, response):
        # type: (HTTPResponse) -> None
        result = {}
        if not response.error:
            result = tornado.escape.json_decode(response.body)
            if result.get('results'):
                result = result['results']
                AnywhereLandingCache.set_country_restrictions(
                    self._from_id, zlib.compress(json.dumps(result).encode('utf-8')))
            else:
                logger.info('No country restrictions in response for city %d', self._from_id)
        else:
            logger.info('Response error: %s', response.error)

        self.finish_callback(result, field=self.field)
