# -*- encoding: utf-8 -*-
import logging
from typing import Optional
from datetime import datetime
from functools import partial

import tornado.escape
from tornado.web import HTTPError

from travel.avia.library.python.iata_correction import IATACorrector

from travel.avia.api_gateway.application.fetcher import Fetcher
from travel.avia.api_gateway.application.fetcher.flight_landing.mapper import FlightLandingMapper
from travel.avia.api_gateway.application.fetcher.conditional_flight_fetcher import get_flight_by_number_fetcher
from travel.avia.api_gateway.application.fetcher.backend import AirlineFetcher

logger = logging.getLogger(__name__)


class FlightLandingFetcher(Fetcher):
    service = 'flight_landing'

    ARRIVAL_DAY_PERIOD = '2:4'
    AIRLINE_FIELDS = ','.join(
        [
            'rating',
            'url',
            'title',
            'baggage_rules_url',
            'baggage_rules',
            'default_tariff',
            'registration_url',
            'baggage',
            'carryon',
        ]
    )

    def __init__(self, flight_landing_mapper, iata_corrector, *args, **kwargs):
        # type: (FlightLandingMapper, IATACorrector, any, any) -> None
        super(FlightLandingFetcher, self).__init__(*args, **kwargs)
        self._flight_landing_mapper = flight_landing_mapper
        self._iata_corrector = iata_corrector

    def fetch(self, fetchers=None):
        company_code = self.params.get('company_code')
        flight_number = self.params.get('flight_number')
        national_version = self.params.get('national_version')
        user_geo_id = self.params.get('user_geo_id')
        lang = self.params.get('lang') or 'ru'
        raw_departure_date = self.params.get('departure_date')
        time_utc = self._get_time_utc(raw_departure_date).isoformat()
        from_code = self.params.get('from_code')

        company = self._get_company(company_code, flight_number)

        fetcher = Fetcher(
            finish_callback=partial(
                self.on_response,
                company_code,
                flight_number,
                user_geo_id,
                raw_departure_date,
                from_code,
                national_version,
                lang,
            )
        )
        fetchers = []

        fetcher.waiting_fields.add('flights')
        fetchers.append(
            get_flight_by_number_fetcher(
                field='flights',
                company_iata=company_code,
                number=flight_number,
                need_map=False,
                params={'time_utc': time_utc, 'arrival_day_period': self.ARRIVAL_DAY_PERIOD},
                cache_root=self.cache_root,
            )
        )

        fetcher.waiting_fields.add('airline_info')
        fetchers.append(
            AirlineFetcher(
                field='airline_info',
                airline_id=company.Id,
                lang=lang,
                fields=self.AIRLINE_FIELDS,
            )
        )

        fetcher.fetch(fetchers)

    def _get_company(self, company_code, flight_number):
        carrier_by_flight_number = self._iata_corrector.flight_numbers_to_carriers([(company_code, flight_number)])
        company_id = carrier_by_flight_number.get('{} {}'.format(company_code, flight_number))
        company = self.cache_root.company_cache.get_company_by_id(company_id)
        if company:
            return company

        raise HTTPError(404, reason='Unknown company_code {}'.format(tornado.escape.url_escape(company_code)))

    def on_response(
        self,
        company_code,
        flight_number,
        user_geo_id,
        raw_departure_date,
        from_code,
        national_version,
        lang,
        data,
    ):
        # type: (str, str, Optional[int], Optional[str], Optional[str], str, str, dict) -> None
        flights = data.get('flights', [])
        if not flights:
            raise HTTPError(
                404,
                reason='No flights with company code = {} and flight number = {}'.format(
                    tornado.escape.url_escape(company_code),
                    tornado.escape.url_escape(flight_number),
                ),
            )
        airline_info = data.get('airline_info', {})
        mapped_landing = self._flight_landing_mapper.map(
            user_geo_id,
            raw_departure_date,
            from_code,
            national_version,
            lang,
            flights,
            airline_info,
        )
        self.finish_callback(mapped_landing)

    def _get_time_utc(self, raw_departure_date):
        # type: (Optional[str]) -> datetime
        if not raw_departure_date:
            return datetime.utcnow()
        try:
            return datetime.strptime(raw_departure_date, '%Y-%m-%d')
        except:
            raise HTTPError(
                400, reason='Invalid departure_date argument: {}'.format(tornado.escape.url_escape(raw_departure_date))
            )
