# -*- encoding: utf-8 -*-
from typing import Dict, List, Optional, Any
from datetime import datetime, timedelta
from collections import namedtuple
import logging

import pytz
from ciso8601 import parse_datetime_as_naive
from tornado.web import HTTPError
from lxml import etree

from travel.avia.api_gateway.application.cache.cache_root import CacheRoot
from travel.avia.api_gateway.lib.landings.templater import LandingTemplater, get_point_title_nominative
from travel.avia.api_gateway.application.utils.format_time import NBSP, human_duration
from travel.avia.api_gateway.application.fetcher.flight_landing.terms import TermMapper
from travel.avia.api_gateway.application.fetcher.flight_landing.enums import (
    Color,
    TermType,
    PlainTextStyle,
    FlightBlockType,
)
from travel.avia.api_gateway.application.cache.landing_routes_cache import LandingRoute

logger = logging.getLogger(__name__)

MSK_TZ = pytz.timezone('Europe/Moscow')
RUSSIA_COUNTRY_ID = 225

MONTH_NAMES = [
    'янв',
    'фев',
    'мар',
    'апр',
    'мая',
    'июн',
    'июл',
    'авг',
    'сен',
    'окт',
    'ноя',
    'дек',
]

FULL_MONTH_NAMES = [
    'января',
    'февраля',
    'марта',
    'апреля',
    'мая',
    'июня',
    'июля',
    'августа',
    'сентября',
    'октября',
    'ноября',
    'декабря',
]

WEEKDAY_NAMES = [
    'пн',
    'вт',
    'ср',
    'чт',
    'пт',
    'сб',
    'вс',
]

FULL_WEEKDAY_NAMES = [
    'понедельник',
    'вторник',
    'среда',
    'четверг',
    'пятница',
    'суббота',
    'воскресенье',
]

XML_TEMPLATE = """<FareDetail>
    <Leg>
        <Seg>
            <FromCountry>{FromCountry}</FromCountry>
            <ToCountry>{ToCountry}</ToCountry>
            <FromAirport>{FromAirport}</FromAirport>
            <ToAirport>{ToAirport}</ToAirport>
        </Seg>
        <CountryOfSale>{CountryOfSale}</CountryOfSale>
    </Leg>
</FareDetail>"""


class FlightLandingMapper:
    MAX_TOP_AIRLINES = 10
    FLIGHT_LANDING_IMAGE_URL = 'https://yastat.net/s3/travel/static/_/images/013730b9706fc64b269fcf4266d4a975.png'
    FLIGHT_LANDING_IMAGE_SIZE = (180, 180)
    BREADCRUMBS_TITLE_LINK = '/avia/'
    ROUTE_LANDING_URL_TEMPLATE = '/avia/routes/{}--{}/'

    def __init__(self, cache_root, templater):
        # type: (CacheRoot, LandingTemplater) -> None
        self._cache_root = cache_root
        self._templater = templater
        self._term_mapper = TermMapper(templater)

    def map(self, user_geo_id, raw_departure_date, from_code, national_version, lang, flights, airline_info):
        # type: (Optional[int], Optional[str], Optional[str], str, str, List[Dict], Dict) -> dict
        user_tz = self.pytz_by_geo_id(user_geo_id)
        now_aware = datetime.now(user_tz)

        flights = [Segment.from_raw(s, now_aware, self._cache_root) for s in flights]
        default_flight = self._select_default_flight(flights, raw_departure_date, now_aware)

        from_station, from_point, to_station, to_point = self._get_points(default_flight)
        from_point_key = self._point_key(from_point)
        to_point_key = self._point_key(to_point)

        default_flight_segments = default_flight.segments or [default_flight]

        user_country = self._get_user_country(user_geo_id)
        return {
            'seoInfo': self._seo_info(from_point, to_point, default_flight),
            'blocks': {
                block.pop('type'): block
                for block in [
                    _f
                    for _f in [
                        self._search_form(
                            from_point,
                            to_point,
                            from_point_key,
                            to_point_key,
                            default_flight.raw_segment['departureDay'],
                        ),
                        self._breadcrumbs_block(from_point, to_point, default_flight, national_version),
                        self._flight_title_block(from_point, to_point, default_flight),
                        self._flight_dates_block(flights, default_flight),
                        self._flight_block(default_flight_segments, from_code, now_aware, lang),
                        self._map_block(default_flight_segments),
                        self._disclaimer_block(),
                        self._additional_info_block(
                            airline_info,
                            default_flight,
                            lang,
                            user_country,
                        ),
                        self._tickets_block(default_flight, now_aware, from_point_key, to_point_key),
                        self._useful_links_block(airline_info),
                        self._flight_seo_text_block(
                            default_flight,
                            from_point,
                            to_point,
                            from_station,
                            to_station,
                        ),
                    ]
                    if _f
                ]
            },
        }

    def pytz_by_geo_id(self, geo_id):
        """
        Возвращает временную зону по указанному geo_id.
        В случае невозможности возвращает default таймзону
        :param geo_id: id геолокации пользователя
        :param default: дефеолтная таймзона для возврата, если не удалось вытянуть из geo_id
        :return:
        """
        if geo_id:
            settlement = self._cache_root.settlement_cache.get_settlement_by_geo_id(geo_id)
            if settlement:
                return self._get_timezone_by_settlement(settlement)
        return MSK_TZ

    def _seo_info(self, from_point, to_point, default_flight):
        flight_title = self._build_flight_title(default_flight)
        company_title = self._get_company_title(default_flight.company)
        title = self._templater.render(
            'seo.title',
            flight_title=flight_title,
            from_point=from_point,
            to_point=to_point,
            company_title=company_title,
        )
        description = self._templater.render(
            'seo.description',
            flight_title=flight_title,
            from_point=from_point,
            to_point=to_point,
            company_title=company_title,
        )
        return {
            'title': title,
            'description': description,
            'openGraph': {
                'title': title,
                'description': description,
                'image': self.FLIGHT_LANDING_IMAGE_URL,
                'imageSize': {
                    'width': self.FLIGHT_LANDING_IMAGE_SIZE[0],
                    'height': self.FLIGHT_LANDING_IMAGE_SIZE[1],
                },
            },
            'schemaOrg': {},
        }

    def _breadcrumbs_block(self, from_point, to_point, default_flight, national_version):
        route_landing_link = None
        if LandingRoute(from_point.Id, to_point.Id, national_version) in self._cache_root.landing_routes_cache:
            route_landing_link = self.ROUTE_LANDING_URL_TEMPLATE.format(
                self._cache_root.settlement_cache.get_slug_by_id(from_point.Id),
                self._cache_root.settlement_cache.get_slug_by_id(to_point.Id),
            )
        return {
            'type': FlightBlockType.BREADCRUMBS_BLOCK,
            'items': [
                {
                    'title': self._templater.render('breadcrumbs.main_title'),
                    'link': self.BREADCRUMBS_TITLE_LINK,
                },
                {
                    'title': self._templater.render(
                        'breadcrumbs.cities_title',
                        from_point=from_point,
                        to_point=to_point,
                    ),
                    'link': route_landing_link,
                },
                {
                    'title': self._templater.render(
                        'breadcrumbs.flight_title',
                        flight_title=self._build_flight_title(default_flight),
                    ),
                },
            ],
        }

    @staticmethod
    def _build_flight_title(flight):
        return ' '.join((flight.company_code, flight.number))

    def _select_default_flight(self, flights, raw_departure_date, now_aware):
        # type: (List[Segment], Optional[str], datetime) -> Segment
        if raw_departure_date:
            flights_on_date = [f for f in flights if f.raw_segment['departureDay'] == raw_departure_date]
            if not flights_on_date:
                raise HTTPError(404, reason='No flights on date {}'.format(raw_departure_date))
            return flights_on_date[0]
        for f in flights:
            if now_aware - f.arrival.get_estimated_time() <= timedelta(hours=2):
                return f
        return flights[0]

    def _flight_title_block(self, from_point, to_point, default_flight):
        company_title = self._get_company_title(default_flight.company)
        return {
            'type': FlightBlockType.FLIGHT_TITLE_BLOCK,
            'title': self._templater.render(
                'flight_title',
                flight_title=self._build_flight_title(default_flight),
                from_point=from_point,
                to_point=to_point,
                company_title=company_title,
            ),
        }

    def _get_points(self, default_flight):
        from_station = default_flight.departure.station
        to_station = default_flight.arrival.station
        from_point = self._get_settlement_by_station(from_station) or from_station
        to_point = self._get_settlement_by_station(to_station) or to_station

        return from_station, from_point, to_station, to_point

    def _get_settlement_by_station(self, station):
        if station.SettlementId:
            settlement = self._cache_root.settlement_cache.get_settlement_by_id(station.SettlementId)
            if settlement:
                return settlement

        settlement_id = self._cache_root.station_to_settlement_cache.get_settlement_id_by_station_id(station.Id)
        if settlement_id:
            return self._cache_root.settlement_cache.get_settlement_by_id(station.SettlementId)

        return None

    def _flight_dates_block(self, flights, default_flight):
        items = [
            {
                'departureDate': f.raw_segment['departureDay'],
                'shownDepartureDate': self._get_shown_date_with_weekday_name(f.raw_segment['departureDay']),
                'selected': f.raw_segment['departureDay'] == default_flight.raw_segment['departureDay'],
            }
            for f in flights
        ]
        default_flight_index = next(i for i, item in enumerate(items) if item['selected'])
        items = items[max(default_flight_index - 1, 0) : default_flight_index + 3]
        return {
            'type': FlightBlockType.FLIGHT_DATES_BLOCK,
            'items': items,
        }

    def _get_shown_date(self, point_datetime):
        # type: (datetime) -> basestring
        return '{day_number} {month_name}'.format(
            day_number=point_datetime.day,
            month_name=MONTH_NAMES[point_datetime.month - 1],
        )

    def _get_shown_date_with_weekday_name(self, raw_departure_date):
        # type: (str) -> basestring
        departure_date = datetime.strptime(raw_departure_date, '%Y-%m-%d')
        return '{day_number} {month_name}, {day_name}'.format(
            day_number=departure_date.day,
            month_name=MONTH_NAMES[departure_date.month - 1],
            day_name=WEEKDAY_NAMES[departure_date.weekday()],
        )

    def _flight_block(self, default_flight_segments, from_code, now_aware, lang):
        # type: (List[Segment], Optional[str], datetime, str) -> Dict
        default_segment = self._select_default_segment(default_flight_segments, from_code, now_aware)
        return {
            'type': FlightBlockType.FLIGHT_BLOCK,
            'segments': [
                {
                    'arrival': self._map_segment_point(s.arrival),
                    'departure': self._map_segment_point(s.departure),
                    'duration': self._get_duration(s, now_aware),
                    'lastUpdate': 'Данные обновлены {} назад'.format(
                        self._human_duration_short(now_aware - s.updated_at)
                    ),
                    'status': s.status_info.to_dict(lang, self._cache_root, self._templater),
                    'selected': s.departure.time == default_segment.departure.time,
                }
                for s in default_flight_segments
            ],
        }

    def _map_segment_point(self, point):
        station_title = get_point_title_nominative(self._cache_root.station_cache.get_station_by_id(point.station.Id))
        settlement_title = get_point_title_nominative(self._get_settlement_by_station(point.station))
        common_title = settlement_title or station_title
        result = {
            'stationTitle': station_title,
            'settlementTitle': settlement_title,
            'title': common_title,
            'stationCode': point.code,
            'formattedDate': self._get_shown_date(point.scheduled_time),
            'gate': point.gate,
            'scheduledTime': self._format_datetime(point.scheduled_time),
            'terminal': point.terminal,
            'actualTime': self._format_datetime(point.time),
        }
        result.update(self._select_segment_point_colors(point))
        return result

    def _search_form(self, from_point, to_point, from_point_key, to_point_key, departure_date):
        return {
            'type': FlightBlockType.SEARCH_FORM_BLOCK,
            'fromTitle': get_point_title_nominative(from_point),
            'toTitle': get_point_title_nominative(to_point),
            'fromId': from_point_key,
            'toId': to_point_key,
            'departureDate': departure_date,
        }

    def _get_duration(self, segment, now_aware):
        duration = segment.arrival.scheduled_time - segment.departure.scheduled_time

        if (
            not segment.departure.is_actual
            or not segment.arrival.is_actual
            or segment.status_info.status == 'unknown'
            or now_aware < segment.departure.time
        ):
            in_air = None
            passed = 0
        elif now_aware < segment.arrival.time:
            in_air = now_aware - segment.departure.time
            passed = min(in_air.total_seconds() / duration.total_seconds(), 1)
        else:
            in_air = None
            passed = 1
        return {
            'total': self._human_duration_short(duration),
            'inAir': self._human_duration_short(in_air) if in_air else None,
            'passed': passed,
        }

    @staticmethod
    def _human_duration_short(delta):
        # type: (timedelta) -> basestring
        minutes = int(delta.total_seconds()) // 60
        days = minutes // 60 // 24
        hours = (minutes // 60) % 24
        minutes = minutes % (60 * 24) % 60

        blocks = []

        if days:
            blocks.append('{} д'.format(days))

        if hours:
            blocks.append('{} ч'.format(hours))

        if minutes:
            blocks.append('{} мин'.format(minutes))

        return ' '.join(b.replace(' ', NBSP) for b in blocks)

    def _select_default_segment(self, segments, from_code, now_aware):
        # type: (List[Segment], Optional[str], datetime) -> Segment
        if from_code:
            segments_with_from_code = [s for s in segments if s.departure.code == from_code]
            if segments_with_from_code:
                return segments_with_from_code[0]
        for s in segments:
            if now_aware - s.arrival.get_estimated_time() <= timedelta(hours=2):
                return s
        return segments[0]

    def _get_point_title(self, point):
        try:
            return point.Title.Ru
        except:
            try:
                return point.Title.Ru.Nominative
            except:
                return point.TitleDefault

    @staticmethod
    def _select_segment_point_colors(fp):
        # type: (SegmentPoint) -> Dict[str, Color]

        def _choose_time_color(fp):
            # type: (SegmentPoint) -> Color
            if fp.time and fp.time > fp.scheduled_time:
                return Color.RED

            if fp.time and fp.time == fp.scheduled_time:
                return Color.BLACK

            return Color.GREY

        def _choose_date_color(fp):
            # type: (SegmentPoint) -> Color

            if fp.time and fp.time.date() > fp.scheduled_time.date():
                return Color.RED

            if fp.time and fp.time.date() == fp.scheduled_time.date():
                return Color.BLACK

            return Color.GREY

        return {
            'timeColor': _choose_time_color(fp),
            'dateColor': _choose_date_color(fp),
            'terminalColor': Color.GREY,
            'gateColor': Color.GREY,
        }

    def _get_timezone_by_settlement(self, settlement):
        tz = self._cache_root.timezone_cache.get_timezone_by_id(settlement.TimeZoneId)
        if tz:
            return pytz_timezone(tz.Code)
        region = self._cache_root.region_cache.get_region_by_id(settlement.RegionId)
        if region:
            tz = self._cache_root.timezone_cache.get_timezone_by_id(region.TimeZoneId)
            if tz:
                return pytz_timezone(tz.Code)
        return MSK_TZ

    def _map_block(self, segments):
        # type: (List[Segment]) -> Dict
        return {
            'type': FlightBlockType.MAP_BLOCK,
            'flightPointCoordinates': (
                [self._station_coordinates(s.departure.station) for s in segments]
                + [self._station_coordinates(segments[-1].arrival.station)]
            ),
        }

    @staticmethod
    def _station_coordinates(station):
        return {'latitude': station.Latitude, 'longitude': station.Longitude}

    def _disclaimer_block(self):
        return {
            'type': FlightBlockType.DISCLAIMER_BLOCK,
            'text': self._templater.render('flight.disclaimer'),
        }

    @staticmethod
    def _format_datetime(dt):
        # type: (Optional[datetime]) -> Optional[str]
        if dt:
            return dt.strftime('%Y-%m-%dT%H:%M')
        return None

    def _point_key(self, point):
        if self._cache_root.settlement_cache.get_settlement_by_id(point.Id):
            return 'c{}'.format(point.Id)
        return 's{}'.format(point.Id)

    def _additional_info_block(self, airline_info, default_flight, lang, user_country):
        additional_info = {
            'type': FlightBlockType.ADDITIONAL_INFO_BLOCK,
        }

        flight_rating_items = self._build_flight_rating_items(airline_info.get('rating'))
        if flight_rating_items:
            additional_info['flightRating'] = {
                'title': self._templater.render('flight_rating.title'),
                'items': self._build_flight_rating_items(airline_info.get('rating')),
            }

        baggage_seo_text = self._build_baggage_seo_text(airline_info)
        if baggage_seo_text:
            additional_info['seoText'] = baggage_seo_text

        aircraft_type = self._build_aircraft_type(default_flight)
        if aircraft_type:
            additional_info['aircraftType'] = aircraft_type

        tariffs = self._build_tariffs(default_flight, lang, user_country)
        if tariffs:
            additional_info['tariffs'] = tariffs
        else:
            baggage_info = self._build_baggage_info(airline_info)
            if baggage_info:
                additional_info['baggage'] = baggage_info

        return additional_info

    def _build_aircraft_type(self, default_flight):
        if not default_flight.transport_model or not default_flight.transport_model.Title:
            return None
        return {
            'title': self._templater.render('aircraft_type.title'),
            'aircraftType': default_flight.transport_model.Title,
        }

    def _build_baggage_info(self, airline_info):
        default_tariff = airline_info.get('default_tariff') or {}

        baggage_info = {
            'title': self._templater.render('baggage_block.title'),
            'description': self._templater.render('baggage_block.description'),
            'baggageRestrictions': {
                'title': self._templater.render('baggage_restrictions.title'),
            },
        }

        def _get_dimensions_string(dimensions):
            width = dimensions.get('width')
            length = dimensions.get('length')
            height = dimensions.get('height')
            if width and length and height:
                return '{}x{}x{}'.format(width, length, height)
            return None

        carryon = airline_info.get('carryon')
        if carryon and default_tariff.get('carryon'):
            carryon_dimensions = _get_dimensions_string(carryon)
            carryon_weight = default_tariff.get('carryon_norm', 0)
            if isinstance(carryon_weight, float) and carryon_weight == int(carryon_weight):
                carryon_weight = int(carryon_weight)
            if carryon_dimensions or carryon_weight:
                baggage_info['baggageRestrictions']['carryOn'] = {
                    'type': TermType.CARRY_ON,
                    'description': self._templater.render(
                        'carryon.description',
                        dimensions=carryon_dimensions,
                        weight=carryon_weight,
                    ),
                    'color': Color.BLACK,
                }

        baggage = airline_info.get('baggage')
        if baggage and default_tariff.get('baggage_allowed'):
            baggage_dimensions = _get_dimensions_string(baggage)
            baggage_weight = default_tariff.get('baggage_norm', 0)
            if isinstance(baggage_weight, float) and baggage_weight == int(baggage_weight):
                baggage_weight = int(baggage_weight)
            if baggage_dimensions or baggage_weight:
                baggage_info['baggageRestrictions']['baggage'] = {
                    'type': TermType.BAGGAGE,
                    'description': self._templater.render(
                        'baggage.description',
                        dimensions=baggage_dimensions,
                        weight=baggage_weight,
                    ),
                    'color': Color.BLACK,
                }
        if not (
            baggage_info['baggageRestrictions'].get('carryOn') or baggage_info['baggageRestrictions'].get('baggage')
        ):
            del baggage_info['baggageRestrictions']
        return baggage_info

    def _build_flight_rating_items(self, rating):
        if not rating:
            return []
        in_time_percent = 100
        rating_items = []
        for key in (
            'canceled',
            'delayedMore90',
            'delayed6090',
            'delayed3060',
            'delayedLess30',
        ):
            if rating.get(key):
                rating_items.append(
                    {
                        'text': self._templater.render('flight_rating.{}'.format(key)),
                        'percent': '{}%'.format(rating[key]),
                    }
                )
                in_time_percent -= rating[key]

        rating_items.append(
            {'text': self._templater.render('flight_rating.inTime'), 'percent': '{}%'.format(in_time_percent)}
        )
        return rating_items[::-1]

    def _tickets_block(self, default_flight, now_aware, from_point_key, to_point_key):
        # type: (Segment, datetime, str, str) -> Optional[Dict]
        if from_point_key == to_point_key:
            return None
        if default_flight.departure.scheduled_time - timedelta(hours=1) < now_aware:
            return None
        departure_date = default_flight.departure.scheduled_time.date()
        day_number = departure_date.day
        month_name = FULL_MONTH_NAMES[departure_date.month - 1]
        day_name = FULL_WEEKDAY_NAMES[departure_date.weekday()]
        return {
            'type': FlightBlockType.TICKETS_BLOCK,
            'text': {
                'title': self._templater.render('tickets.title'),
                'text': [
                    {
                        'type': FlightBlockType.TEXT_BLOCK,
                        'children': [
                            {
                                'type': FlightBlockType.PLAIN_TEXT_BLOCK,
                                'data': {
                                    'text': '{day_number} {month_name}, {day_name}, 1 {passenger}'.format(
                                        day_number=day_number,
                                        month_name=month_name,
                                        day_name=day_name,
                                        passenger=self._templater.render('tickets.passenger'),
                                    ),
                                    'styles': [PlainTextStyle.BOLD],
                                },
                            },
                        ],
                    },
                    {
                        'type': FlightBlockType.TEXT_BLOCK,
                        'children': [
                            {
                                'type': FlightBlockType.PLAIN_TEXT_BLOCK,
                                'data': {
                                    'text': self._templater.render('tickets.baggage_class'),
                                },
                            },
                        ],
                    },
                ],
            },
            'button': {
                'text': self._templater.render('tickets.button'),
                'link': self._build_tickets_link(default_flight, from_point_key, to_point_key),
            },
        }

    def _build_tickets_link(self, default_flight, from_point_key, to_point_key):
        return (
            '/avia/order/?adult_seats=1&backward=&children_seats=0&klass=economy&oneway=1&return_date=&infant_seats=0&'
            + 'forward={code} {flight_number}.{departure_datetime}&'.format(
                code=default_flight.company_code,
                flight_number=default_flight.number,
                departure_datetime=self._format_datetime(default_flight.departure.scheduled_time),
            )
            + 'fromId={from_point_key}&toId={to_point_key}&when={when}'.format(
                from_point_key=from_point_key,
                to_point_key=to_point_key,
                when=default_flight.raw_segment['departureDay'],
            )
            + '#empty'
        )

    def _useful_links_block(self, airline_info):
        items = []
        for key in ('url', 'baggage_rules_url', 'registration_url'):
            if airline_info.get(key):
                text = self._templater.render('useful_links.{}'.format(key))
                url = airline_info.get(key)
                if text:
                    items.append(
                        {
                            'type': FlightBlockType.EXTERNAL_LINK_BLOCK,
                            'data': {
                                'text': text,
                                'url': url,
                            },
                        }
                    )
        return {
            'type': FlightBlockType.USEFUL_LINKS_BLOCK,
            'title': self._templater.render('useful_links.title'),
            'items': items,
        }

    def _flight_seo_text_block(self, default_flight, from_point, to_point, from_station, to_station):
        duration = default_flight.arrival.scheduled_time - default_flight.departure.scheduled_time
        duration_in_minutes = int(duration.total_seconds()) // 60
        from_terminal = default_flight.departure.terminal
        to_terminal = default_flight.arrival.terminal
        company_title = self._get_company_title(default_flight.company)
        return {
            'type': FlightBlockType.FLIGHT_SEO_TEXT,
            'title': self._templater.render(
                'flight_seo_text.title',
                company_code=default_flight.company_code,
                number=default_flight.number,
                from_point=from_point,
                to_point=to_point,
            ),
            'description': self._templater.render(
                'flight_seo_text.extended_description',
                company_code=default_flight.company_code,
                company_title=company_title,
                number=default_flight.number,
                from_point=from_point if from_point != from_station else None,
                to_point=to_point if to_point != to_station else None,
                from_station_title=from_station.Title.Ru,
                to_station_title=to_station.Title.Ru,
                from_terminal=from_terminal,
                to_terminal=to_terminal,
                duration=human_duration(duration_in_minutes),
            ),
        }

    def _build_baggage_seo_text(self, airline_info):
        baggage_rules = airline_info.get('baggage_rules')
        if not baggage_rules:
            return None
        return {
            'type': FlightBlockType.SECTION_TEXT_BLOCK,
            'data': {},
            'children': [
                {
                    'type': FlightBlockType.TEXT_BLOCK,
                    'children': [
                        {
                            'type': FlightBlockType.PLAIN_TEXT_BLOCK,
                            'data': {
                                'text': baggage_rules,
                            },
                        },
                    ],
                }
            ],
        }

    def _build_tariffs(self, flight, lang, user_country):
        fare_families = self._cache_root.fare_family_cache.get_fare_families_by_airline_id(flight.company.Id)
        if not fare_families:
            return None

        flight_xml_description = self._build_flight_xml_description(flight, user_country)

        items = []
        tariff_titles = set()
        for f in fare_families:
            terms = self._get_fare_family_terms(f, flight_xml_description, lang)
            title = f.TariffGroupName.Values.get(lang)
            if title and terms and title not in tariff_titles:
                tariff_titles.add(title)
                items.append(
                    {
                        'title': title,
                        'items': terms,
                    }
                )

        if not items:
            return None
        return {
            'title': self._templater.render('tariffs.title'),
            'items': items,
        }

    def _get_user_country(self, user_geo_id):
        settlement = self._cache_root.settlement_cache.get_settlement_by_geo_id(user_geo_id)
        if not settlement or not settlement.CountryId:
            return self._cache_root.country_cache.get_country_by_id(RUSSIA_COUNTRY_ID)
        return self._cache_root.country_cache.get_country_by_id(settlement.CountryId)

    def _build_flight_xml_description(self, flight, user_country):
        from_country = self._cache_root.country_cache.get_country_by_id(flight.departure.station.CountryId)
        to_country = self._cache_root.country_cache.get_country_by_id(flight.arrival.station.CountryId)
        flight_xml_description = XML_TEMPLATE.format(
            FromCountry=self._get_country_code(from_country),
            ToCountry=self._get_country_code(to_country),
            FromAirport=flight.departure.code,
            ToAirport=flight.arrival.code,
            CountryOfSale=self._get_country_code(user_country),
        )
        return etree.fromstring(flight_xml_description)

    def _get_country_code(self, country):
        return country.Code if country else ''

    def _get_fare_family_terms(self, fare_family, flight_xml_description, lang):
        terms = []
        term_types = set()
        for term in fare_family.Terms:
            for rule in term.Rules:
                if self._check_rule(rule, flight_xml_description):
                    mapped_term = self._term_mapper.map_term(term, rule, lang)
                    if mapped_term:
                        term_types.add(mapped_term.get('type'))
                        terms.append(mapped_term)
                    break
        if TermType.REFUNDABLE not in term_types:
            terms.append(self._term_mapper.get_unavailable_refundable())
        if TermType.CHANGING_CARRIAGE not in term_types:
            terms.append(self._term_mapper.get_unavailable_changing_carriage())
        return terms

    def _check_rule(self, rule, flight_xml_description):
        if not rule.Xpath:
            return True
        return bool(flight_xml_description.xpath(self._get_utf8(rule.Xpath)))

    def _get_utf8(self, xpath_text):
        try:
            return xpath_text.decode('utf-8')
        except:
            return xpath_text

    def _get_company_title(self, company):
        title = company.Title
        if company.TitleRu:
            title = company.TitleRu
        return title


def _settlement_id_to_point_key(_id):
    return 'c{}'.format(_id)


class SegmentPoint(
    namedtuple(
        'SegmentPoint',
        ['timezone', 'station', 'code', 'scheduled_time', 'time', 'terminal', 'gate', 'is_actual', 'data_source'],
    )
):
    def get_estimated_time(self):
        # type: () -> datetime

        return self.time or self.scheduled_time

    @classmethod
    def from_raw(
        cls,
        raw_flight,
        tz,
        airport,
        code,
        scheduled_time,
        time,
        terminal,
        gate,
        data_source,
        now_aware,
        cache_root,
    ):
        # type: (Dict[str, Any], str, str, str, str, str, str, str, str, datetime, CacheRoot) -> SegmentPoint

        timezone = pytz_timezone(raw_flight[tz])
        scheduled = pytz.UTC.localize(parse_datetime_as_naive(raw_flight[scheduled_time])).astimezone(timezone)
        station = cache_root.station_cache.get_station_by_id(raw_flight[airport])
        if not station:
            raise Exception('Not found station for %r' % raw_flight)

        actual_time = raw_flight['status'][time]
        actual_time = timezone.localize(parse_datetime_as_naive(actual_time)) if actual_time else None

        gate = raw_flight['status'][gate] or None
        if not _is_actual_departure_info(now_aware, actual_time or scheduled, 20):
            gate = None

        return cls(
            timezone=timezone,
            station=station,
            code=raw_flight[code],
            scheduled_time=scheduled,
            time=actual_time,
            terminal=raw_flight['status'][terminal] or None,
            gate=gate,
            # если shared-flights вернул время в ['status']['time'], считаем, что данные по времени актуальны
            is_actual=bool(actual_time),
            data_source=raw_flight['status'][data_source],
        )


class SegmentStatusInfo(
    namedtuple(
        'FlightStatusInfo',
        [
            'raw_status',
            'departs_soon',
            'departure_is_known',
            'arrival_is_known',
            'baggage_carousels',
            'check_in_desks',
            'diverted',
            'diverted_airport_code',
            'diverted_airport_id',
        ],
    )
):
    UNKNOWN = 'unknown'
    CANCELLED = 'cancelled'
    ARRIVED = 'arrived'
    DELAYED = 'delayed'
    DIVERTED = 'diverted'
    EARLY = 'early'
    ON_TIME = 'on_time'

    _color_by_status = {
        UNKNOWN: Color.YELLOW,
        CANCELLED: Color.RED,
        ARRIVED: Color.GREY,
        DELAYED: Color.RED,
        DIVERTED: Color.RED,
        EARLY: Color.RED,
        ON_TIME: Color.GREEN,
    }

    @property
    def status(self):
        return self.raw_status

    def cancelled(self):
        # type: () -> bool

        return self.status == self.CANCELLED

    def to_dict(self, lang, cache_root, templater):
        # type: (str, CacheRoot, LandingTemplater) -> Dict[str, Any]

        flight_status = {
            'code': self.raw_status,
            'text': self._get_text(lang, cache_root, templater),
            'departsSoon': self.departs_soon,
            'checkInDesks': self._format_check_in_desks(),
            'baggageCarousels': self.baggage_carousels,
        }

        flight_status.update(self._select_flight_status_colors())

        return flight_status

    def _format_check_in_desks(self):
        comma = ','
        if not self.check_in_desks or comma not in self.check_in_desks:
            return self.check_in_desks
        comma_space = ', '
        return comma_space.join(self.check_in_desks.split(comma))

    def _get_text(self, lang, cache_root, templater):
        # type: (str, CacheRoot, LandingTemplater) -> str
        station_id = self.diverted_airport_id
        if self.diverted and station_id:
            station = cache_root.station_cache.get_station_by_id(station_id)
            if station:
                return templater.render(
                    'flight.status.diverted',
                    lang,
                    station=station,
                    station_code=self.diverted_airport_code.upper(),
                )
        if lang == 'ru' and self.status == 'unknown':
            if not self.departure_is_known and not self.arrival_is_known:
                return templater.render('flight_status.no_info', lang)
            if not self.departure_is_known:
                return templater.render('flight_status.no_departure_info', lang)
            if not self.arrival_is_known:
                return templater.render('flight_status.no_arrival_info', lang)

        if self.status in self._color_by_status:
            return templater.render('flight_status.{}'.format(self.status), lang)
        return templater.render('flight_status.unknown', lang)

    def _select_flight_status_colors(self):
        # type: () -> Dict[str, Color]

        return {
            'textColor': self._color_by_status.get(self.status, Color.GREY),
            'checkInDesksColor': Color.GREY,
            'baggageCarouselsColor': Color.GREY,
        }


class Segment(
    namedtuple(
        'Segment',
        [
            'number',
            'company_code',
            'company',
            'transport_model',
            'departure',
            'arrival',
            'status_info',
            'updated_at',
            'raw_segment',
            'segments',
        ],
    )
):
    @classmethod
    def from_raw(cls, raw_segment, now_aware, cache_root):
        # type: (Dict[str, Any], datetime, CacheRoot) -> Segment

        point_from = SegmentPoint.from_raw(
            raw_segment,
            tz='departureTimezone',
            airport='airportFromID',
            code='airportFromCode',
            scheduled_time='departureUtc',
            time='departure',
            terminal='departureTerminal',
            gate='departureGate',
            data_source='departureSource',
            now_aware=now_aware,
            cache_root=cache_root,
        )
        point_to = SegmentPoint.from_raw(
            raw_segment,
            tz='arrivalTimezone',
            airport='airportToID',
            code='airportToCode',
            scheduled_time='arrivalUtc',
            time='arrival',
            terminal='arrivalTerminal',
            gate='arrivalGate',
            data_source='arrivalSource',
            now_aware=now_aware,
            cache_root=cache_root,
        )

        status = raw_segment['status']['status']
        departure_is_known = bool(raw_segment.get('status', {}).get('departure'))
        arrival_is_known = bool(raw_segment.get('status', {}).get('arrival'))
        departs_soon = now_aware > point_from.get_estimated_time() - timedelta(hours=24)

        baggage_carousels = raw_segment['status'].get('baggageCarousels')
        check_in_desks = raw_segment['status'].get('checkInDesks')
        diverted = bool(raw_segment['status'].get('diverted'))
        diverted_airport_code = raw_segment['status'].get('divertedAirportCode')
        diverted_airport_id = raw_segment['status'].get('divertedAirportID')

        segment_status_info = SegmentStatusInfo(
            status,
            departs_soon,
            departure_is_known,
            arrival_is_known,
            baggage_carousels,
            check_in_desks,
            diverted,
            diverted_airport_code,
            diverted_airport_id,
        )

        updated_at = pytz_timezone('UTC').localize(
            parse_datetime_as_naive(max(raw_segment['updatedAtUtc'], raw_segment['status']['updatedAtUtc']))
        )
        company = cache_root.company_cache.get_company_by_id(raw_segment['airlineID'])
        company_code = company.Iata
        if not company_code and company.SirenaId:
            company_code = company.SirenaId

        return cls(
            number=raw_segment['number'],
            company_code=company_code,
            company=company,
            transport_model=cls._get_transport_model(cache_root, raw_segment),
            departure=point_from,
            arrival=point_to,
            status_info=segment_status_info,
            updated_at=updated_at,
            raw_segment=raw_segment,
            segments=[Segment.from_raw(s, now_aware, cache_root) for s in raw_segment.get('segments', [])],
        )

    @staticmethod
    def _get_transport_model(cache_root, raw_segment):
        transport_model_id = raw_segment.get('transportModelID')
        if not transport_model_id:
            return None
        return cache_root.transport_model_cache.get_transport_model_by_id(transport_model_id)


def pytz_timezone(tz_name, _memo={}):
    try:
        return _memo[tz_name]
    except KeyError:
        res = pytz.timezone(tz_name)
        _memo[tz_name] = res
        return res


def _is_actual_departure_info(now_aware, estimated_time, minutes_after_departure):
    return now_aware <= estimated_time + timedelta(minutes=minutes_after_departure)
