# -*- encoding: utf-8 -*-

from travel.avia.api_gateway.application.fetcher.flight_landing.enums import TermType, Color, Availability
from travel.avia.api_gateway.lib.landings.templater import LandingTemplater

ALL_TERM_CODES = {
    'baggage',
    'carry_on',
    'changing_carriage',
    'changing_carriage_no_show',
    'disclosure_url',
    'miles',
    'open_return_date',
    'refundable',
    'refundable_no_show',
    'seat_selection_check_in',
}

TERM_CODE_TO_TERM_TYPE = {
    'baggage': TermType.BAGGAGE,
    'carry_on': TermType.CARRY_ON,
    'changing_carriage': TermType.CHANGING_CARRIAGE,
    'miles': TermType.MILES,
    'refundable': TermType.REFUNDABLE,
    'seat_selection_check_in': TermType.SEAT_SELECTION_CHECK_IN,
}


class TermMapper:
    def __init__(self, templater):
        # type: (LandingTemplater) -> None
        self._templater = templater

    def _map_carry_on(self, term, rule, lang):
        if rule.Places and rule.Weight and rule.Size:
            return {
                'type': TermType.CARRY_ON,
                'description': self._templater.render(
                    'tariffs.carryon_description',
                    places=rule.Places,
                    weight=rule.Weight,
                    size=rule.Size,
                ),
                'color': Color.BLACK,
            }

    def _map_baggage(self, term, rule, lang):
        if rule.Places and rule.Weight:
            return {
                'type': TermType.BAGGAGE,
                'description': self._templater.render(
                    'tariffs.baggage_description',
                    places=rule.Places,
                    weight=rule.Weight,
                    size=rule.Size,
                ),
                'color': Color.BLACK,
            }

    def _map_changing_carriage(self, term, rule, lang):
        if rule.Availability == Availability.NOT_AVAILABLE:
            return self.get_unavailable_changing_carriage()
        if rule.Availability == Availability.FREE:
            return {
                'type': TermType.CHANGING_CARRIAGE,
                'description': self._templater.render('tariffs.changing_carriage_free'),
                'color': Color.GREEN,
            }
        if rule.Availability == Availability.CHARGE:
            return {
                'type': TermType.CHANGING_CARRIAGE,
                'description': self._templater.render('tariffs.changing_carriage_charge'),
                'color': Color.BLACK,
            }

    def _map_refundable(self, term, rule, lang):
        if rule.Availability == Availability.NOT_AVAILABLE:
            return self.get_unavailable_refundable()
        if rule.Availability == Availability.FREE:
            return {
                'type': TermType.REFUNDABLE,
                'description': self._templater.render('tariffs.refundable_free'),
                'color': Color.GREEN,
            }
        if rule.Availability == Availability.CHARGE:
            return {
                'type': TermType.REFUNDABLE,
                'description': self._templater.render('tariffs.refundable_charge'),
                'color': Color.BLACK,
            }

    def _map_seat_selection(self, term, rule, lang):
        if rule.Availability == Availability.CHARGE:
            return {
                'type': TermType.SEAT_SELECTION_CHECK_IN,
                'description': self._templater.render('tariffs.seat_selection_charge'),
                'color': Color.BLACK,
            }
        if rule.Availability == Availability.FREE:
            return {
                'type': TermType.SEAT_SELECTION_CHECK_IN,
                'description': self._templater.render('tariffs.seat_selection_free'),
                'color': Color.GREEN,
            }

    def _map_miles(self, term, rule, lang):
        try:
            if rule.Miles:
                return {
                    'type': TermType.BAGGAGE,
                    'description': self._templater.render('tariffs.miles_description', miles=rule.Miles),
                    'color': Color.GREEN,
                }
        except:
            return None

    def get_unavailable_refundable(self):
        return {
            'type': TermType.REFUNDABLE,
            'description': self._templater.render('tariffs.refundable_unavailable'),
            'color': Color.BLACK,
        }

    def get_unavailable_changing_carriage(self):
        return {
            'type': TermType.CHANGING_CARRIAGE,
            'description': self._templater.render('tariffs.changing_carriage_unavailable'),
            'color': Color.BLACK,
        }

    def map_term(self, term, rule, lang):
        if term.Code not in TERM_CODE_TO_TERM_TYPE:
            return None
        if not rule:
            return None
        term_type = TERM_CODE_TO_TERM_TYPE[term.Code]
        if term_type == TermType.CARRY_ON:
            return self._map_carry_on(term, rule, lang)
        elif term_type == TermType.BAGGAGE:
            return self._map_baggage(term, rule, lang)
        elif term_type == TermType.MILES:
            return self._map_miles(term, rule, lang)
        elif term_type == TermType.SEAT_SELECTION_CHECK_IN:
            return self._map_seat_selection(term, rule, lang)
        elif term_type == TermType.CHANGING_CARRIAGE:
            return self._map_changing_carriage(term, rule, lang)
        elif term_type == TermType.REFUNDABLE:
            return self._map_refundable(term, rule, lang)
