from travel.avia.api_gateway.application.fetcher import Fetcher
from travel.avia.api_gateway.application.fetcher.backend import (
    AirportsFetcher,
    AirlineFetcher,
    FlightRatingFetcher,
    ReviewsFetcher,
)

FLIGHT_EXTRAS_DICT = {
    'aircraft': None,
    'ife_business': None,
    'ife_comfort': None,
    'ife_economy': None,
    'ife_first_class': None,
    'power_business': None,
    'power_comfort': None,
    'power_economy': None,
    'power_first_class': None,
    'seat_pitch_business': None,
    'seat_pitch_comfort': None,
    'seat_pitch_economy': None,
    'seat_pitch_first_class': None,
    'seats_business_class': None,
    'seats_comfort': None,
    'seats_economy': None,
    'seats_first_class': None,
    'seats_total': None,
    'wifi_business': None,
    'wifi_comfort': None,
    'wifi_economy': None,
    'wifi_first_class': None,
}

FLIGHT_EXTRAS_AIRCRAFT_DICT = {
    'code': None,
    'is_cargo': None,
    'producer': None,
    'plane_body_type': None,
    'title': None,
    'title_en': None,
    'is_propeller_flight': None,
    'code_en': None,
}


class _SegmentFetcher(Fetcher):
    def add_result(self, data, field=None):
        # type: (dict, str) -> None

        if field.startswith('segment.airline') and self.result and 'flight' in self.result:
            if data:
                for s in self.result['flight'].get('segments'):
                    if s['flight']['airline_id'] == data.get('id'):
                        s['airline'] = data
            self.add_waiting_request(field)
            return

        if field.startswith('segment.airport') and self.result and 'flight' in self.result:
            if data:
                for s in self.result['flight'].get('segments'):
                    if s['flight']['airport_from_id'] == data.get('id'):
                        s['airport_from'] = data
                    if s['flight']['airport_to_id'] == data.get('id'):
                        s['airport_to'] = data
            self.add_waiting_request(field)
            return

        super(_SegmentFetcher, self).add_result(data, field)


def _prepare_segments(flight, params):
    for s in flight.get('segments'):
        prepared_segment = {
            'flight': s,
            'airline': None,
            'airport_from': None,
            'airport_to': None,
        }

        if 'flight_status' in params.get('fields', []):
            prepared_segment['flight_status'] = s.pop('status', None)

        yield prepared_segment


def _prepare_segments_fetchers(fetcher, flight, params):
    segment_airports = {}

    for s in flight.get('segments'):
        k = 'segment.airline.{}'.format(s['flight']['airline_id'])

        if k not in fetcher.waiting_requests:
            fetcher.waiting_requests.add(k)
            yield AirlineFetcher(
                field=k,
                airline_id=s['flight']['airline_id'],
                lang=params.get('lang'),
            )

        k = 'segment.airport.{}'.format(s['flight']['airport_from_id'])
        if k not in fetcher.waiting_requests:
            fetcher.waiting_requests.add(k)
            segment_airports[k] = s['flight']['airport_from_id']

        k = 'segment.airport.{}'.format(s['flight']['airport_to_id'])
        if k not in fetcher.waiting_requests:
            fetcher.waiting_requests.add(k)
            segment_airports[k] = s['flight']['airport_to_id']

    if segment_airports:
        yield AirportsFetcher(
            airports=segment_airports,
            lang=params.get('lang'),
        )


def _prepare_flight_fetchers(fetcher, flight, params):
    fetcher.waiting_fields.add('airport_from')
    fetcher.waiting_fields.add('airport_to')
    yield AirportsFetcher(
        airports={
            'airport_from': flight.get('airport_from_id'),
            'airport_to': flight.get('airport_to_id'),
        },
        lang=params.get('lang'),
    )

    fetcher.waiting_fields.add('airline')
    yield AirlineFetcher(
        field='airline',
        airline_id=flight.get('airline_id'),
        lang=params.get('lang'),
    )


def _prepare_flight_opt_fetchers(fetcher, flight, params):
    if 'flight_rating' in params.get('fields', []):
        fetcher.waiting_fields.add('flight_rating')
        yield FlightRatingFetcher(
            field='flight_rating',
            flight_number='-'.join(flight.get('number').split(' ')),
        )

    if 'flight_reviews' in params.get('fields', []):
        fetcher.waiting_fields.add('flight_reviews')
        yield ReviewsFetcher(
            field='flight_reviews',
            airline_id=flight.get('airline_id'),
            flight_number=flight.get('number').split(' ')[-1],
        )


def flight_supplement(data, field, params, finish_callback, cache_root):
    fetcher = _SegmentFetcher(field=field, finish_callback=finish_callback, **params)
    flight = data['flight']

    if flight.get('segments'):
        flight['segments'] = list(_prepare_segments(flight, params))

    fetcher.result = {
        field: flight,
    }

    if 'flight_status' in params.get('fields', []):
        fetcher.result['flight_status'] = flight.pop('status', None)

    if 'flight_extras' in params.get('fields', []):
        fetcher.result['flight_extras'] = flight_extras(flight, cache_root)

    fetchers = list(_prepare_flight_fetchers(fetcher, flight, params))

    if flight.get('segments'):
        fetchers.extend(_prepare_segments_fetchers(fetcher, flight, params))

    fetchers.extend(_prepare_flight_opt_fetchers(fetcher, flight, params))

    fetcher.fetch(fetchers)


def flight_extras(flight, cache_root):
    result = FLIGHT_EXTRAS_DICT.copy()
    if not flight['transport_model_id']:
        return result

    model = cache_root.transport_model_cache.get_transport_model_by_id(flight['transport_model_id'])
    if not model:
        return result

    aircraft = FLIGHT_EXTRAS_AIRCRAFT_DICT.copy()
    aircraft['code'] = model.Code
    aircraft['code_en'] = model.CodeEn
    aircraft['is_cargo'] = model.Cargo
    aircraft['is_propeller_flight'] = model.PropellerFlight
    aircraft['plane_body_type'] = model.PlaneBodyType
    aircraft['producer'] = model.ProducerTitle
    aircraft['title'] = model.Title
    aircraft['title_en'] = model.TitleEn
    result['aircraft'] = aircraft

    return result
