# -*- encoding: utf-8 -*-
import tornado.escape
from tornado.httpclient import HTTPResponse
from tornado.httputil import HTTPHeaders

from travel.avia.api_gateway.application.fetcher import Fetcher
from travel.avia.api_gateway.settings import TRAVEL_API_URL, TRAVEL_API_TVM_ID
from travel.avia.api_gateway.application.utils.tvm import tvm_factory
from travel.avia.library.python.urls.search import C


HOTEL_CROSSLINKS_TOP_LIMIT = 7
REQUEST_HEADERS_TO_TRAVEL_API = {
    'X-Ya-Session-Key',
    'X-Ya-YandexUid',
    'X-Ya-User-Ticket',
    'X-Ya-PassportId',
    'X-Ya-User-Ip',
    'X-Ya-UseCamelCase',
    # 'X-Ya-User-Agent' cannot be passed normally because it may contain non-acsii symbols
    # and pycurl (in tornado) produces exceptions in this case
}


class HotelCrosslinksDataFetcher(Fetcher):
    def __init__(self, request_headers, fault_tolerance=False, *args, **kwargs):
        # type: (HotelCrosslinksDataFetcher, HTTPHeaders, any, any) -> None
        super(HotelCrosslinksDataFetcher, self).__init__(*args, **kwargs)
        self._request_headers = request_headers
        self._fault_tolerance = fault_tolerance

    def fetch(self, fetchers=None):  # type: (List[Fetcher]) -> None
        to_point_key = str(C(self.params.get('to_id')))
        national_version = self.params.get('national_version')

        tvm_provider = tvm_factory.get_provider()
        headers = {
            'Content-Type': 'application/json',
            'X-Ya-Service-Ticket': tvm_provider.get_ticket(TRAVEL_API_TVM_ID),
        }
        for name in REQUEST_HEADERS_TO_TRAVEL_API:
            if name in self._request_headers:
                headers[name] = self._request_headers[name]

        self.request(
            f'{TRAVEL_API_URL}/hotels_portal/v1/get_cross_sale_hotels/',
            params={
                'settlement_id': to_point_key,
                'total_hotel_limit': HOTEL_CROSSLINKS_TOP_LIMIT,
                'domain': national_version,
            },
            headers=headers,
            callback=self.on_response,
            raise_error=not self._fault_tolerance,
        )

    def on_response(self, response):
        # type: (HTTPResponse) -> None
        result = {}
        if not response.error:
            result = tornado.escape.json_decode(response.body)
        self.finish_callback(result, field=self.field)
