# -*- encoding: utf-8 -*-
import logging
from datetime import datetime
from typing import Dict
from functools import partial

from tornado.web import HTTPError

from travel.avia.api_gateway.application.fetcher import Fetcher
from travel.avia.api_gateway.application.fetcher.price_index import TopDirectionsByDateWindow

logger = logging.getLogger(__name__)


class IsPossibleTripFetcher(Fetcher):
    service = 'is_possible_trip'

    DATE_FORMAT = '%Y-%m-%d'

    def fetch(self, fetchers=None):
        from_settlement_code = self.params.get('from_settlement_code')
        to_settlement_code = self.params.get('to_settlement_code')
        departure_date = self.params.get('departure_date')
        try:
            departure_date = datetime.strptime(departure_date, self.DATE_FORMAT).strftime(self.DATE_FORMAT)
        except:
            raise HTTPError(400, reason='Invalid date format: {}'.format(departure_date))

        national_version = self.params.get('national_version')
        window_size = self.params.get('window_size')

        from_settlement = self._get_settlement(from_settlement_code)
        to_settlement = self._get_settlement(to_settlement_code)

        if not from_settlement or not to_settlement:
            message_parts = []
            if not from_settlement:
                message_parts.append('No settlement for code {}'.format(from_settlement_code))
            if not to_settlement:
                message_parts.append('No settlement for code {}'.format(to_settlement_code))
            raise HTTPError(404, reason='. '.join(message_parts))

        from_id = from_settlement.Id
        to_id = to_settlement.Id

        request = {
            'directions': [{'from_id': from_id, 'to_id': to_id}],
            'forward_date': departure_date,
            'window_size': window_size,
            'results_per_direction': 1,
        }
        fetcher = Fetcher(finish_callback=partial(self.on_response, from_id, to_id, departure_date))
        fetcher.fetch(
            [
                TopDirectionsByDateWindow(
                    national_version=national_version,
                    request=request,
                    request_timeout=1000,
                )
            ]
        )

    def on_response(self, from_id, to_id, departure_date, price_index_response):
        # type: (int, int, str, Dict) -> None
        if not price_index_response or price_index_response.get('status') != 'ok':
            status = price_index_response.get('status')
            raise HTTPError(500, reason='price-index has responded with status {}'.format(status))

        is_possible_trip = self._is_possible_trip(from_id, to_id, departure_date, price_index_response)
        result = {
            'fromId': from_id,
            'toId': to_id,
            'isPossibleTrip': is_possible_trip,
        }
        if is_possible_trip:
            result['minPrice'] = price_index_response['data'][0]['min_price']
        self.finish_callback(result, field=self.field)

    def _get_settlement(self, settlement_code):
        settlement_code = settlement_code.upper()
        by_iata = self.cache_root.settlement_cache.get_settlement_by_iata(settlement_code)
        if by_iata:
            return by_iata
        by_sirena = self.cache_root.settlement_cache.get_settlement_by_sirena_id(settlement_code)
        if by_sirena:
            return by_sirena
        station_id = self.cache_root.station_code_cache.get_station_id_by_code(settlement_code)
        station = self.cache_root.station_cache.get_station_by_id(station_id)
        if station and station.SettlementId:
            return self.cache_root.settlement_cache.get_settlement_by_id(station.SettlementId)
        try:
            settlement_id = int(settlement_code)
        except:
            raise HTTPError(400, reason='Invalid settlement code {}'.format(settlement_code))

        return self.cache_root.settlement_cache.get_settlement_by_id(
            settlement_id
        ) or self.cache_root.settlement_cache.get_settlement_by_geo_id(settlement_id)

    def _is_possible_trip(self, from_id, to_id, departure_date, price_index_response):
        data = price_index_response.get('data')
        return bool(
            data
            and len(data) == 1
            and data[0].get('min_price')
            and data[0]['from_id'] == from_id
            and data[0]['to_id'] == to_id
            and data[0]['forward_date'] == departure_date
        )
