# -*- encoding: utf-8 -*-
import logging
from typing import Dict

from tornado.web import HTTPError

from travel.avia.api_gateway.lib import feature_flags
from travel.avia.api_gateway.application.fetcher import Fetcher
from travel.avia.api_gateway.application.fetcher.route_landing.mapper import RouteLandingMapper
from travel.avia.api_gateway.application.fetcher.avia_statistics import RouteLandingDataFetcher
from travel.avia.api_gateway.application.fetcher.hotel_crosslinks import HotelCrosslinksDataFetcher

logger = logging.getLogger(__name__)


class RouteLandingFetcher(Fetcher):
    service = 'route_landing'

    def __init__(self, route_landing_mapper, request_headers, *args, **kwargs):
        # type: (RouteLandingMapper, Dict, any, any) -> None
        super(RouteLandingFetcher, self).__init__(*args, **kwargs)
        self._route_landing_mapper = route_landing_mapper
        self._request_headers = request_headers

    def fetch(self, fetchers=None):
        from_slug = self.params.get('from_slug')
        to_slug = self.params.get('to_slug')
        national_version = self.params.get('national_version')

        from_settlement = self.cache_root.settlement_cache.get_settlement_by_slug(from_slug)
        to_settlement = self.cache_root.settlement_cache.get_settlement_by_slug(to_slug)

        if not from_settlement or not to_settlement or not national_version:
            message_parts = []
            if not from_settlement:
                message_parts.append('No settlement for from_slug {}'.format(from_slug))
            if not to_settlement:
                message_parts.append('No settlement for to_slug {}'.format(to_slug))
            raise HTTPError(404, reason='. '.join(message_parts))

        from_id = from_settlement.Id
        to_id = to_settlement.Id

        logger.info('Getting route landing for from_id: %s, to_id: %s, nv: %s', from_id, to_id, national_version)
        if feature_flags.enable_hotels_block_in_routes_landing():
            fetcher = Fetcher(self.on_response)
            fetcher.waiting_fields = {'avia', 'hotel_crosslinks'}
            fetcher.fetch(
                [
                    RouteLandingDataFetcher(
                        field='avia',
                        from_id=from_id,
                        to_id=to_id,
                        national_version=national_version,
                    ),
                    HotelCrosslinksDataFetcher(
                        request_headers=self._request_headers,
                        fault_tolerance=True,
                        field='hotel_crosslinks',
                        to_id=to_id,
                        national_version=national_version,
                    )
                ]
            )
        else:
            fetcher = Fetcher(self.on_response)
            fetcher.fetch(
                [
                    RouteLandingDataFetcher(
                        from_id=from_id,
                        to_id=to_id,
                        national_version=national_version,
                    )
                ]
            )

    def on_response(self, response):
        # type: (Dict) -> None
        if feature_flags.enable_hotels_block_in_routes_landing():
            mapped_landing = self._route_landing_mapper.map(response['avia'], response['hotel_crosslinks'])
        else:
            mapped_landing = self._route_landing_mapper.map(response, {})
        self.finish_callback(mapped_landing, field=self.field)
