# -*- encoding: utf-8 -*-
from enum import Enum
from datetime import date, timedelta
import copy
import logging

from travel.avia.api_gateway.application.cache.cache_root import CacheRoot
from travel.avia.api_gateway.lib.landings.templater import LandingTemplater
from travel.avia.api_gateway.application.utils import human_duration, MONTHS
from travel.avia.api_gateway.settings import TRAVEL_FRONTEND_URL_RU
from travel.avia.library.python.enum.currency import Currency

logger = logging.getLogger(__name__)


class RouteLandingMapper:
    MAX_TOP_AIRLINES = 10
    ROUTE_LANDING_IMAGE_URL = 'https://avatars.mds.yandex.net/get-avia/233213/2a0000015a80437fec5bb0af3293c5f3b4b9/orig'
    ROUTE_LANDING_OPENGRAPH_IMAGE_URL = (
        'https://yastat.net/s3/travel/static/_/images/013730b9706fc64b269fcf4266d4a975.png'
    )
    ROUTE_LANDING_OPENGRAPH_IMAGE_SIZE = (180, 180)
    BREADCRUMBS_TITLE_LINK = '{}/avia/'.format(TRAVEL_FRONTEND_URL_RU)

    def __init__(self, cache_root, templater):
        # type: (CacheRoot, LandingTemplater) -> None
        self._cache_root = cache_root
        self._templater = templater

    def map(self, landing, hotel_crosslinks):
        # type: (dict, dict) -> dict
        if not landing:
            return {}
        from_settlement = self._cache_root.settlement_cache.get_settlement_by_id(landing['fromId'])
        to_settlement = self._cache_root.settlement_cache.get_settlement_by_id(landing['toId'])
        from_point_key = _settlement_id_to_point_key(from_settlement.Id)
        to_point_key = _settlement_id_to_point_key(to_settlement.Id)
        top_airlines = landing.get('topAirlines')
        airline_id, min_price = self._get_airline_with_min_price(top_airlines)
        airline_with_min_price = self._cache_root.company_cache.get_company_by_id(airline_id)
        faq_data = self._build_faq_data(
            from_settlement,
            to_settlement,
            landing.get('routeInfo'),
            top_airlines,
            airline_with_min_price,
            min_price,
        )
        faq_question_answers = self._build_faq_question_answers(faq_data)
        return {
            'seoInfo': self._seo_info(from_settlement, to_settlement, min_price, faq_question_answers),
            'blocks': [
                _f
                for _f in [
                    self._breadcrumbs_block(from_settlement, to_settlement),
                    self._search_form(from_settlement, to_settlement, from_point_key, to_point_key),
                    self._dynamics_block(from_settlement, to_settlement, from_point_key, to_point_key),
                    self._best_to_fly_block(
                        from_settlement,
                        to_settlement,
                        landing.get('medianPrices'),
                        landing.get('popularMonth'),
                    ),
                    self._airlines_block(from_settlement, to_settlement, top_airlines, with_transfers=False),
                    self._airlines_block(from_settlement, to_settlement, top_airlines, with_transfers=True),
                    self._closest_cities_block(
                        to_settlement,
                        landing.get('alternativeRoutesPrices'),
                    ),
                    self._faq_block(faq_data, faq_question_answers),
                    self._return_ticket_block(
                        from_settlement,
                        to_settlement,
                        landing.get('returnTicketPrice'),
                    ),
                    self._crosslinks_block(landing.get('routeCrosslinks')),
                    self._hotel_crosslinks_block(hotel_crosslinks, to_settlement)
                ]
                if _f
            ],
        }

    def _seo_info(self, from_settlement, to_settlement, min_price, faq_question_answers):
        title = self._templater.render(
            'seo.title',
            city_from=from_settlement,
            city_to=to_settlement,
            min_price=min_price,
        )
        description = self._templater.render(
            'seo.description',
            city_from=from_settlement,
            city_to=to_settlement,
        )
        return {
            'title': title,
            'description': description,
            'openGraph': {
                'title': title,
                'description': description,
                'image': self.ROUTE_LANDING_OPENGRAPH_IMAGE_URL,
                'imageSize': {
                    'width': self.ROUTE_LANDING_OPENGRAPH_IMAGE_SIZE[0],
                    'height': self.ROUTE_LANDING_OPENGRAPH_IMAGE_SIZE[1],
                },
            },
            'schemaOrg': {
                'faqItems': [
                    {
                        'question': q,
                        'answer': a,
                    }
                    for q, a in faq_question_answers
                ],
            },
        }

    def _breadcrumbs_block(self, from_settlement, to_settlement):
        return {
            'type': ERouteBlockType.BREADCRUMBS_BLOCK,
            'data': {
                'items': [
                    {
                        'title': self._templater.render('breadcrumbs.main_title'),
                        'link': self.BREADCRUMBS_TITLE_LINK,
                    },
                    {
                        'title': self._templater.render(
                            'breadcrumbs.cities_title',
                            city_from=from_settlement,
                            city_to=to_settlement,
                        ),
                    },
                ]
            },
        }

    def _search_form(self, from_settlement, to_settlement, from_point_key, to_point_key):
        return {
            'type': ERouteBlockType.SEARCH_FORM_BLOCK,
            'data': {
                'title': self._templater.render(
                    'search_form.title',
                    city_from=from_settlement,
                    city_to=to_settlement,
                ),
                'imageUrl': self.ROUTE_LANDING_IMAGE_URL,
                'searchFormParams': {
                    'isCalendarOpen': True,
                    'fromId': from_point_key,
                    'toId': to_point_key,
                },
            },
        }

    def _dynamics_block(self, from_settlement, to_settlement, from_point_key, to_point_key):
        return {
            'type': ERouteBlockType.DYNAMICS_BLOCK,
            'data': {
                'title': self._templater.render('dynamic.title', city_from=from_settlement, city_to=to_settlement),
                'navigationTitle': self._templater.render('dynamic.navigation'),
                'description': self._templater.render(
                    'dynamic.description',
                    city_from=from_settlement,
                    city_to=to_settlement,
                ),
                'fromId': from_point_key,
                'toId': to_point_key,
                'when': (date.today() + timedelta(days=3)).strftime('%Y-%m-%d'),
            },
        }

    def _best_to_fly_block(self, from_settlement, to_settlement, median_prices_info, popular_month_info):
        if not median_prices_info:
            return None
        title = self._templater.render(
            'best_to_fly.title',
            city_from=from_settlement,
            city_to=to_settlement,
        )
        children = [
            _f
            for _f in [
                self._min_price_month_block(median_prices_info),
                self._popular_month_block(popular_month_info, median_prices_info['yearMedianPrice']),
            ]
            if _f
        ]
        if not children:
            return None
        return {
            'type': ERouteBlockType.SECTION_TEXT_BLOCK,
            'data': {
                'title': title,
                'navigationTitle': self._templater.render('best_to_fly.navigation'),
            },
            'children': children,
        }

    def _min_price_month_block(self, median_prices_info):
        min_price = median_prices_info['monthMedianPrice']
        year_median_price = median_prices_info['yearMedianPrice']

        if not min_price or not year_median_price:
            return None

        if min_price > year_median_price:
            min_price_delta = int((year_median_price / min_price - 1) * 100)
        else:
            min_price_delta = int((year_median_price - min_price) * 100.0 / year_median_price)
        min_price_month_text = self._templater.render(
            'best_to_fly.min_price_month',
            min_price_month=MONTHS[median_prices_info['month'] - 1].nominative,
            min_price=min_price,
            year_median_price=year_median_price,
            price_delta=min_price_delta,
        )
        return {
            'type': ERouteBlockType.TEXT_BLOCK,
            'children': [
                {
                    'type': ERouteBlockType.PLAIN_TEXT_BLOCK,
                    'data': {
                        'text': min_price_month_text,
                    },
                },
            ],
        }

    def _popular_month_block(self, popular_month_info, year_median_price):
        if not popular_month_info or not year_median_price:
            return None
        popular_month_price = popular_month_info['price']
        popular_month_text = self._templater.render(
            'best_to_fly.popular_month',
            popular_month=MONTHS[popular_month_info['month'] - 1].nominative.capitalize(),
            popular_month_price=popular_month_price,
            year_median_price=year_median_price,
            price_delta=price_delta(popular_month_price, year_median_price),
        )
        return {
            'type': ERouteBlockType.TEXT_BLOCK,
            'children': [
                {
                    'type': ERouteBlockType.PLAIN_TEXT_BLOCK,
                    'data': {
                        'text': popular_month_text,
                    },
                },
            ],
        }

    def _extract_title_and_min_price(self, top_airlines, with_transfers):
        price_field = 'priceWithTransfers' if with_transfers else 'price'
        company_title = None
        min_price = None
        airlines = [a for a in top_airlines if a.get(price_field)]
        if airlines:
            airline_price = min(airlines, key=lambda a: a[price_field])
            company = self._cache_root.company_cache.get_company_by_id(airline_price['id'])
            if company:
                company_title = company.TitleRu
                min_price = airline_price[price_field]
        return company_title, min_price

    def _airlines_block(self, from_settlement, to_settlement, top_airlines, with_transfers):
        if not top_airlines:
            return None

        min_price_company_title, min_price = self._extract_title_and_min_price(top_airlines, with_transfers)
        if not min_price:
            return None

        block_key = 'indirect_airlines' if with_transfers else 'direct_airlines'
        return {
            'type': ERouteBlockType.AVIACOMPANIES_BLOCK,
            'data': {
                'title': self._templater.render(
                    '{}.title'.format(block_key),
                    city_from=from_settlement,
                    city_to=to_settlement,
                ),
                'navigationTitle': self._templater.render('{}.navigation'.format(block_key)),
                'description': self._templater.render(
                    '{}.description'.format(block_key),
                    min_price_company_title=min_price_company_title,
                    min_price=min_price,
                ),
                'items': [
                    {
                        'id': str(a['id']),
                        'name': self._cache_root.company_cache.get_company_by_id(a['id']).TitleRu,
                        'when': a.get('departureDateWithTransfers') if with_transfers else a.get('departureDate'),
                        'price': {
                            'value': a.get('priceWithTransfers') if with_transfers else a.get('price'),
                            'currency': Currency.from_str_with_correction(a['currency']),
                        },
                        'withTransfers': with_transfers,
                    }
                    for a in top_airlines[: self.MAX_TOP_AIRLINES]
                    if with_transfers and a.get('priceWithTransfers') or not with_transfers and a.get('price')
                ],
            },
        }

    def _closest_cities_block(self, to_settlement, alternative_routes_prices):
        if not alternative_routes_prices:
            return None
        return {
            'type': ERouteBlockType.CLOSEST_CITIES_BLOCK,
            'data': {
                'title': self._templater.render('closest.title', city_to=to_settlement),
                'navigation': self._templater.render('closest.navigation'),
                'description': self._templater.render('closest.description'),
                'items': [
                    {
                        'toSlug': self._cache_root.settlement_cache.get_slug_by_id(r['alternativeToId']),
                        'name': self._cache_root.settlement_cache.get_settlement_by_id(
                            r['alternativeToId']
                        ).Title.Ru.Nominative,
                        'when': r['date'],
                        'price': {
                            'value': r['price'],
                            'currency': Currency.from_str_with_correction(r['currency']),
                        },
                    }
                    for r in alternative_routes_prices
                ],
            },
        }

    def _build_faq_question_answers(self, faq_data):
        part_number = 1
        question_answers = []
        while True:
            question = self._templater.render('faq.{}_q'.format(part_number), data=faq_data)
            answer = self._templater.render('faq.{}_a'.format(part_number), data=faq_data)
            if question is None or answer is None:
                return question_answers
            part_number += 1
            if not answer:
                continue
            question_answers.append((question, answer))

    def _faq_block(self, faq_data, faq_question_answers):
        faq_parts = []
        for question, answer in faq_question_answers:
            faq_parts.append(
                {
                    'type': ERouteBlockType.SPOILER_BLOCK,
                    'data': {
                        'title': question,
                        'description': {
                            'type': ERouteBlockType.TEXT_BLOCK,
                            'children': [
                                {
                                    'type': ERouteBlockType.PLAIN_TEXT_BLOCK,
                                    'data': {
                                        'text': answer,
                                    },
                                },
                            ],
                        },
                    },
                }
            )

        return {
            'type': ERouteBlockType.SECTION_TEXT_BLOCK,
            'data': {
                'title': self._templater.render('faq.title', data=faq_data),
                'navigationTitle': self._templater.render('faq.navigation'),
            },
            'children': faq_parts,
        }

    def _build_airport_titles(self, airport_ids):
        return ', '.join(self._cache_root.station_cache.get_station_by_id(_id).TitleDefault for _id in airport_ids)

    @staticmethod
    def _get_airline_with_min_price(top_airlines):
        if not top_airlines:
            return None, None
        airlines_with_price = [a for a in top_airlines if a.get('price') or a.get('priceWithTransfers')]
        if not airlines_with_price:
            return None, None

        min_price_getter = lambda airline: min(
            [_f for _f in (airline.get('price'), airline.get('priceWithTransfers')) if _f]
        )
        airline_with_min_price = min(
            airlines_with_price,
            key=min_price_getter,
        )
        return airline_with_min_price['id'], min_price_getter(airline_with_min_price)

    def _build_faq_data(
        self,
        from_settlement,
        to_settlement,
        route_info,
        top_airlines,
        airline_with_min_price,
        min_price,
    ):
        from_airport_titles = self._build_airport_titles(route_info['fromAirports'])
        to_airport_titles = self._build_airport_titles(route_info['toAirports'])
        top_airlines = top_airlines or []
        direct_airline_titles = ', '.join(
            self._cache_root.company_cache.get_company_by_id(a['id']).TitleRu for a in top_airlines if a.get('price')
        )
        airline_with_min_price_title = airline_with_min_price.Title if airline_with_min_price else None
        return FaqData(
            from_settlement,
            to_settlement,
            direct_airline_titles=direct_airline_titles,
            airline_with_min_price_title=airline_with_min_price_title,
            min_price=min_price,
            from_airport_titles=from_airport_titles,
            to_airport_titles=to_airport_titles,
            route_duration_minutes=route_info.get('duration'),
            route_duration=human_duration(route_info.get('duration')),
            route_distance=route_info.get('distance'),
        )

    def _return_ticket_block(self, from_settlement, to_settlement, return_ticket):
        if not return_ticket:
            return None
        return {
            'type': ERouteBlockType.RETURN_TICKET_BLOCK,
            'data': {
                'title': self._templater.render(
                    'return_ticket.title',
                    city_from=from_settlement,
                    city_to=to_settlement,
                ),
                'navigationTitle': self._templater.render('return_ticket.navigation'),
                'items': [
                    {
                        'fromSlug': self._cache_root.settlement_cache.get_slug_by_id(to_settlement.Id),
                        'toSlug': self._cache_root.settlement_cache.get_slug_by_id(from_settlement.Id),
                        'name': '{} – {}'.format(
                            to_settlement.Title.Ru.Nominative,
                            from_settlement.Title.Ru.Nominative,
                        ),
                        'price': {
                            'value': return_ticket['price'],
                            'currency': Currency.from_str_with_correction(return_ticket['currency']),
                        },
                    }
                ],
            },
        }

    def _crosslinks_block(self, crosslinks):
        if not crosslinks:
            return None

        def map_price(crosslink):
            if not (crosslink.get('price') and crosslink.get('currency')):
                return None
            return {
                'value': crosslink['price'],
                'currency': Currency.from_str_with_correction(crosslink['currency']),
            }

        get_settlement_by_id = self._cache_root.settlement_cache.get_settlement_by_id

        return {
            'type': ERouteBlockType.RETURN_TICKET_BLOCK,
            'data': {
                'title': self._templater.render('crosslinks.title'),
                'navigationTitle': self._templater.render('crosslinks.navigation'),
                'items': [
                    {
                        'fromSlug': self._cache_root.settlement_cache.get_slug_by_id(c.get('crosslinkFromId')),
                        'toSlug': self._cache_root.settlement_cache.get_slug_by_id(c.get('crosslinkToId')),
                        'name': '{} – {}'.format(
                            get_settlement_by_id(c.get('crosslinkFromId')).Title.Ru.Nominative,
                            get_settlement_by_id(c.get('crosslinkToId')).Title.Ru.Nominative,
                        ),
                        'price': map_price(c),
                    }
                    for c in crosslinks
                ],
            },
        }

    def _hotel_crosslinks_block(self, hotel_crosslinks, to_settlement):
        if not hotel_crosslinks or 'hasData' not in hotel_crosslinks or not hotel_crosslinks['hasData']:
            return None

        data = copy.copy(hotel_crosslinks)
        data['title'] = self._templater.render('hotel_crosslinks.title', city_to=to_settlement)
        return {
            'type': ERouteBlockType.HOTELS_CROSS_SALE,
            'data': data,
        }


class FaqData:
    def __init__(
        self,
        from_settlement,
        to_settlement,
        direct_airline_titles,
        airline_with_min_price_title,
        min_price,
        from_airport_titles,
        to_airport_titles,
        route_duration_minutes,
        route_duration,
        route_distance,
    ):
        self.city_from = from_settlement
        self.city_to = to_settlement
        self.direct_airline_titles = direct_airline_titles
        self.airline_with_min_price_title = airline_with_min_price_title
        self.min_price = min_price
        self.from_airport_titles = from_airport_titles
        self.to_airport_titles = to_airport_titles
        self.route_duration_minutes = route_duration_minutes
        self.route_duration = route_duration
        self.route_distance = route_distance


def price_delta(month_price, year_price):
    if month_price > year_price:
        return int((round(float(month_price) / year_price - 1, 2)) * 100)
    return int(round((year_price - month_price) * 100.0 / year_price, 2))


def _settlement_id_to_point_key(_id):
    return 'c{}'.format(_id)


class ERouteBlockType(str, Enum):
    BREADCRUMBS_BLOCK = 'IBreadCrumbsBlock'
    SEARCH_FORM_BLOCK = 'ISearchFormBlock'
    TEXT_BLOCK = 'ITextBlock'
    SECTION_TEXT_BLOCK = 'ISectionTextBlock'
    PLAIN_TEXT_BLOCK = 'IPlainTextBlock'
    DYNAMICS_BLOCK = 'IDynamicsBlock'
    AVIACOMPANIES_BLOCK = 'IAviaCompaniesBlock'
    CLOSEST_CITIES_BLOCK = 'IClosestCitiesBlock'
    SPOILER_BLOCK = 'ISpoilerBlock'
    RETURN_TICKET_BLOCK = 'IReturnTicketBlock'
    HOTELS_CROSS_SALE = 'ICrossSaleHotelsBlock'
