# -*- encoding: utf-8 -*-
import logging
from functools import partial
from typing import Dict, Any

from tornado.web import HTTPError

from travel.avia.api_gateway.application.fetcher import Fetcher
from travel.avia.api_gateway.application.fetcher.avia_statistics import RouteLandingDataFetcher
from travel.avia.api_gateway.lib.model_utils import get_settlement_by_code

logger = logging.getLogger(__name__)


class SlugsByRouteFetcher(Fetcher):
    service = 'slugs_by_route'

    def __init__(self, *args, **kwargs):
        # type: (any, any) -> None
        super(SlugsByRouteFetcher, self).__init__(*args, **kwargs)

    def fetch(self, fetchers=None):
        from_settlement_code = self.params.get('from_settlement_code')
        to_settlement_code = self.params.get('to_settlement_code')
        national_version = self.params.get('national_version')

        try:
            from_settlement = get_settlement_by_code(self.cache_root, from_settlement_code)
            to_settlement = get_settlement_by_code(self.cache_root, to_settlement_code)
        except BaseException as err:
            raise HTTPError(404, err)

        self._check_settlements(from_settlement, to_settlement, from_settlement_code, to_settlement_code)

        from_id = from_settlement.Id
        to_id = to_settlement.Id

        slugs_data = {
            'from': self._get_slug_by_settlement(from_settlement),
            'to': self._get_slug_by_settlement(to_settlement),
        }
        fetcher = Fetcher(partial(self.finish_fetch, slugs_data))
        fetcher.fetch(
            [
                RouteLandingDataFetcher(
                    from_id=from_id,
                    to_id=to_id,
                    national_version=national_version,
                )
            ]
        )

    def finish_fetch(self, slugs_data, response):
        # type: (Any, Any, Dict) -> None
        self.finish_callback(slugs_data, field=self.field)

    def _get_slug_by_settlement(self, settlement):
        slug = self.cache_root.settlement_cache.get_slug_by_id(settlement.Id)
        if not slug:
            raise HTTPError(404, reason='No slug for settlement id {}'.format(settlement.Id))
        return slug

    @staticmethod
    def _check_settlements(from_settlement, to_settlement, from_settlement_code, to_settlement_code):
        if not (from_settlement and to_settlement):
            unknown_codes = []
            if not from_settlement:
                unknown_codes.append(from_settlement_code)
            if not to_settlement:
                unknown_codes.append(to_settlement_code)
            raise HTTPError(404, reason='No settlements have been found for codes {}'.format(', '.join(unknown_codes)))
