# coding: utf-8
import logging
import os

from jaeger_client.constants import SAMPLER_TYPE_PROBABILISTIC
from tornado.web import Application
from tornado_opentracing_helpers import setup_tracing
from tornado import ioloop

from travel.avia.library.python.iata_correction import IATACorrector
from travel.avia.library.python.shared_dicts.cache.company_cache import CompanyCache
from travel.avia.library.python.shared_dicts.cache.country_cache import CountryCache
from travel.avia.library.python.shared_dicts.cache.fare_family_cache import FareFamilyCache
from travel.avia.library.python.shared_dicts.cache.region_cache import RegionCache
from travel.avia.library.python.shared_dicts.cache.settlement_cache import SettlementCache
from travel.avia.library.python.shared_dicts.cache.station_cache import StationCache
from travel.avia.library.python.shared_dicts.cache.station_code_cache import StationCodeCache
from travel.avia.library.python.shared_dicts.cache.station_to_settlement_cache import StationToSettlementCache
from travel.avia.library.python.shared_dicts.cache.timezone_cache import TimezoneCache
from travel.avia.library.python.shared_dicts.cache.transport_model_cache import TransportModelCache
from travel.avia.library.python.tanker.client import TankerClient

from travel.avia.api_gateway.application.cache.cache_root import CacheRoot
from travel.avia.api_gateway.application.cache.landing_routes_cache import LandingRoutesCache
from travel.avia.api_gateway.application.cache.tanker_text_cache import (
    RouteLandingTextCache,
    FlightLandingTextCache,
    CityToLandingTextCache,
)
from travel.avia.api_gateway.lib.landings.templater import LandingTemplater

from travel.avia.api_gateway.application.fetcher.anywhere_landing.fetcher import init_settlements_images_cache
from travel.avia.api_gateway.application.handlers import (
    AirlinesSynonymsHandler,
    AirportFlightListHandler,
    AirportTabloSourceHandler,
    AnywhereLandingHandler,
    CityToLandingHandler,
    CountryLandingHandler,
    CurrencyHandler,
    IsPossibleTripHandler,
    FlightByDepartureDateHandler,
    FlightByNumberHandler,
    FlightExtrasByDepartureDateHandler,
    FlightLandingHandler,
    GeoPointHandler,
    MailFlightByDepartureDateHandler,
    NearDirectionsHandler,
    PartnersPopularByRouteHandler,
    PersonalSearchHandler,
    PingHandler,
    RouteLandingHandler,
    SettlementSlugByGeoIdHandler,
    SettlementSlugByRouteHandler,
    ShutdownHandler,
    StatHandler,
    TopFlightsHandler,
    TravelRecipesHandler,
    VersionHandler,
)
from travel.avia.api_gateway.settings import (
    setup_logging,
    TANKER_API_URL,
    SANDBOX_OAUTH_TOKEN,
    SHARED_FLIGHTS_API_URL,
    TANKER_OAUTH_TOKEN,
    TANKER_BRANCH_ID,
)

VERSION = os.getenv('DEPLOY_DOCKER_IMAGE', ':unknown').rsplit(':', 1)[-1]
TRACING_SERVICE_NAME = 'avia-api-gateway'
TRACING_JAEGER_SAMPLER_TYPE = os.getenv('JAEGER_SAMPLER_TYPE', SAMPLER_TYPE_PROBABILISTIC)
TRACING_JAEGER_SAMPLER_PARAMETER = float(os.getenv('JAEGER_SAMPLER_PARAMETER', 0.001))

DISABLE_UPDATING_CACHES = bool(int(os.getenv('DISABLE_UPDATING_CACHES', 0)))


def create_application(shutdown_flag, debug=False):
    setup_logging()

    logger = logging.getLogger()
    if debug:
        logger.setLevel(logging.DEBUG)

    env_variables = ['DEPLOY_STAGE_ID', 'DEPLOY_NODE_DC', 'DEPLOY_BOX_ID']

    tornado_application_tracing_kwargs = setup_tracing(
        service_name=TRACING_SERVICE_NAME,
        version=VERSION,
        include_env_variables=env_variables,
        traced_attributes_of_tornado_request=['method', 'path', 'query', 'query_arguments', 'body_arguments'],
        filter_paths=['/ping', '/unistat'],
        jaeger_sampler_type=TRACING_JAEGER_SAMPLER_TYPE,
        jaeger_sampler_parameter=TRACING_JAEGER_SAMPLER_PARAMETER,
    )

    application = Application(
        [
            (r'/ping', PingHandler),
            (r'/version', VersionHandler),
            (r'/unistat', StatHandler),
            (r'/shutdown', ShutdownHandler),
            (r'/airport/([^/]+)/flight/', AirportFlightListHandler),
            (r'/airport/tablo-source', AirportTabloSourceHandler),
            (r'/currency/', CurrencyHandler),
            (r'/flight/([^/]+)/([^/]+)/(\d{4}\-\d{2}\-\d{2})/', FlightByDepartureDateHandler),
            (r'/flight/([^/]+)/([^/]+)/', FlightByNumberHandler),
            (r'/flight/mail/', MailFlightByDepartureDateHandler),
            (r'/flight-extras/flight/([^/]+)/([^/]+)/(\d{4}\-\d{2}\-\d{2})/', FlightExtrasByDepartureDateHandler),
            (r'/airlines/synonyms/', AirlinesSynonymsHandler),
            (r'/partners/popular-by-route/', PartnersPopularByRouteHandler),
            (r'/geo/point/', GeoPointHandler),
            (r'/travel/recipes/', TravelRecipesHandler),
            (r'/travel/near-directions/', NearDirectionsHandler),
            (r'/top-flights/', TopFlightsHandler),
            (r'/landing/route/([a-z-]+)/([a-z-]+)/([a-z]+)/([a-z]+)/', RouteLandingHandler),
            (r'/landing/city-to/([^/]+)/([a-z]+)/([a-z]+)/', CityToLandingHandler),
            (r'/landing/flight/([^/]+)/([^/]+)/([^/]+)/([^/]+)/', FlightLandingHandler),
            (r'/landing/country/', CountryLandingHandler),
            (r'/landing/anywhere/', AnywhereLandingHandler),
            (r'/settlement-slug-by-geo-id/', SettlementSlugByGeoIdHandler),
            (r'/settlement-slugs-by-route/', SettlementSlugByRouteHandler),
            (r'/is-possible-trip/([^/]+)/([^/]+)/([^/]+)/([^/]+)/', IsPossibleTripHandler),
            (r'/personal-search/', PersonalSearchHandler),
        ],
        debug=debug,
        **tornado_application_tracing_kwargs
    )

    setup_caches(application, logger)

    application.iata_corrector = IATACorrector(shared_flights_api_base_url=SHARED_FLIGHTS_API_URL)

    application.shutdown_flag = shutdown_flag
    return application


def setup_caches(application, logger):
    station_code_cache = StationCodeCache(logger)
    station_cache = StationCache(logger)
    settlement_cache = SettlementCache(logger)
    station_to_settlement_cache = StationToSettlementCache(logger, SANDBOX_OAUTH_TOKEN)
    region_cache = RegionCache(logger)
    country_cache = CountryCache(logger)
    timezone_cache = TimezoneCache(logger)
    company_cache = CompanyCache(logger, SANDBOX_OAUTH_TOKEN)
    fare_family_cache = FareFamilyCache(logger, SANDBOX_OAUTH_TOKEN)
    transport_model_cache = TransportModelCache(logger, SANDBOX_OAUTH_TOKEN)

    tanker = TankerClient(TANKER_API_URL, TANKER_OAUTH_TOKEN)
    city_to_landing_text_cache = CityToLandingTextCache(tanker, logger, TANKER_BRANCH_ID)
    route_landing_text_cache = RouteLandingTextCache(tanker, logger, TANKER_BRANCH_ID)
    flight_landing_text_cache = FlightLandingTextCache(tanker, logger, TANKER_BRANCH_ID)

    landing_routes_cache = LandingRoutesCache(logger)

    caches = [
        station_code_cache,
        settlement_cache,
        company_cache,
        city_to_landing_text_cache,
        flight_landing_text_cache,
        route_landing_text_cache,
        station_cache,
        region_cache,
        timezone_cache,
        fare_family_cache,
        country_cache,
        station_to_settlement_cache,
        transport_model_cache,
        landing_routes_cache,
    ]
    update_period = 24 * 60 * 60 * 1000
    for cache in caches:
        cache.populate()
        if DISABLE_UPDATING_CACHES:
            continue
        ioloop.PeriodicCallback(cache.update_cache, update_period).start()

    application.cache_root = CacheRoot(
        station_code_cache,
        settlement_cache,
        station_to_settlement_cache,
        company_cache,
        city_to_landing_text_cache,
        route_landing_text_cache,
        station_cache,
        flight_landing_text_cache,
        region_cache,
        country_cache,
        timezone_cache,
        fare_family_cache,
        transport_model_cache,
        landing_routes_cache,
    )
    application.city_to_landing_templater = LandingTemplater(city_to_landing_text_cache)
    application.route_landing_templater = LandingTemplater(route_landing_text_cache)
    application.flight_landing_templater = LandingTemplater(flight_landing_text_cache)
    init_settlements_images_cache()
