# -*- encoding: utf-8 -*-
from datetime import datetime
from urllib.parse import urlencode
from urllib.parse import urljoin

import pytz
from tornado.web import HTTPError


class MailMapper:
    def __init__(self, avia_frontend_url_ru):
        self.AVIA_FRONTEND_URL = avia_frontend_url_ru

    def mail_flight_by_departure_date(self, source):
        return self.flight_by_departure_date(source)

    def flight_by_departure_date(self, source):
        data = {
            # Константы
            't_type': 'plane',
            't_type_id': 2,
            # Оставляем пустыми
            'title': None,
            'plain_title': None,
            'from_phrase': None,
            'from_name_gen': None,
            'terminal_name': None,
            'title_parts': [],
            'link': None,
            'mobile_link': None,
            'company_id': None,
            'number': None,
            'scheduled_departure': None,
            'departure_tz': None,
            'scheduled_arrival': None,
            'arrival_tz': None,
            'from_id': None,
            'to_id': None,
            'company': None,
            'company_iata': None,
            'company_sirena': None,
            'from_name': None,
            'from_name_popular': None,
            'from_prefix': None,
            'from_iata': None,
            'from_sirena': None,
            'from_country_geoid': None,
            'from_settlement_geoid': None,
            'to_name': None,
            'to_name_popular': None,
            'to_prefix': None,
            'to_iata': None,
            'to_sirena': None,
            'to_country_geoid': None,
            'to_settlement_geoid': None,
            'status': None,
            'status_title': None,
            'departure': None,
            'arrival': None,
        }

        flight = source.get('flight', {})

        # Half flight from shared-flights
        if not flight.get('arrival_utc'):
            raise HTTPError(404, 'Half flight')

        data.update(
            {
                'company_id': flight.get('airline_id'),
                'number': flight.get('number'),
                'scheduled_departure': self.get_datetime(
                    datetime.strptime(flight.get('departure_utc'), '%Y-%m-%d %H:%M:%S'),
                    flight.get('departure_timezone'),
                ).strftime('%Y-%m-%d %H:%M:%S %z'),
                'departure_tz': flight.get('departure_timezone'),
                'scheduled_arrival': self.get_datetime(
                    datetime.strptime(flight.get('arrival_utc'), '%Y-%m-%d %H:%M:%S'), flight.get('arrival_timezone')
                ).strftime('%Y-%m-%d %H:%M:%S %z'),
                'arrival_tz': flight.get('arrival_timezone'),
                'from_id': flight.get('airport_from_id'),
                'to_id': flight.get('airport_to_id'),
            }
        )

        airline = source.get('airline', {})
        data.update(
            {
                'company': airline.get('title'),
                'company_iata': airline.get('iata'),
                'company_sirena': airline.get('sirena'),
            }
        )

        airport_from = source.get('airport_from', {})
        if airport_from:
            data.update(
                {
                    'from_name': airport_from.get('title'),
                    'from_name_popular': airport_from.get('popularTitle'),
                    'from_prefix': airport_from.get('stationType', {}).get('prefix'),
                    'from_iata': airport_from.get('iataCode'),
                    'from_sirena': airport_from.get('sirenaCode'),
                    'from_country_geoid': airport_from.get('settlement', {}).get('country', {}).get('geoId'),
                    'from_settlement_geoid': airport_from.get('settlement', {}).get('geoId'),
                }
            )

        airport_to = source.get('airport_to', {})
        if airport_to:
            data.update(
                {
                    'to_name': airport_to.get('title'),
                    'to_name_popular': airport_to.get('popularTitle'),
                    'to_prefix': airport_to.get('stationType', {}).get('prefix'),
                    'to_iata': airport_to.get('iataCode'),
                    'to_sirena': airport_to.get('sirenaCode'),
                    'to_country_geoid': airport_to.get('settlement', {}).get('country', {}).get('geoId'),
                    'to_settlement_geoid': airport_to.get('settlement', {}).get('geoId'),
                }
            )

        status = source.get('flight_status', {})

        data['status'] = status.get('status')
        data['status_title'] = {
            'unknown': 'Информация о статусе недоступна',
            'cancelled': 'Отменён',
            'arrived': 'Прилетел',
            'delayed': 'Задержан',
            'early': 'Вылет раньше',
            'on_time': 'По расписанию',
        }.get(data['status'])

        if status.get('departure'):
            data['departure'] = self.localize_datetime(
                datetime.strptime(status.get('departure'), '%Y-%m-%d %H:%M:%S'),
                flight.get('departure_timezone'),
            ).strftime('%Y-%m-%d %H:%M:%S %z')
        elif 'departure_utc' in flight:
            data['departure'] = self.get_datetime(
                datetime.strptime(flight.get('departure_utc'), '%Y-%m-%d %H:%M:%S'), flight.get('departure_timezone')
            ).strftime('%Y-%m-%d %H:%M:%S %z')

        if status.get('arrival'):
            data['arrival'] = self.localize_datetime(
                datetime.strptime(status.get('arrival'), '%Y-%m-%d %H:%M:%S'),
                flight.get('arrival_timezone'),
            ).strftime('%Y-%m-%d %H:%M:%S %z')
        elif 'arrival_utc' in flight:
            data['arrival'] = self.get_datetime(
                datetime.strptime(flight.get('arrival_utc'), '%Y-%m-%d %H:%M:%S'), flight.get('arrival_timezone')
            ).strftime('%Y-%m-%d %H:%M:%S %z')

        if flight.get('number') and flight.get('departure_day'):
            flight_number_for_link = '-'.join(flight.get('number').split())
            link_params = [
                ('when', flight.get('departure_day')),
                ('utm_source', 'yandex-mail'),
            ]
            data['link'] = data['mobile_link'] = urljoin(
                self.AVIA_FRONTEND_URL,
                'flights/{}/?{}'.format(
                    flight_number_for_link,
                    urlencode(link_params),
                ),
            )

        return data

    @staticmethod
    def get_datetime(utc, tz_name):
        # type: (datetime, basestring) -> datetime
        return pytz.timezone('UTC').localize(utc).astimezone(pytz.timezone(tz_name))

    @staticmethod
    def localize_datetime(naive_dt, tz_name):
        # type: (datetime, basestring) -> datetime
        return pytz.timezone(tz_name).localize(naive_dt)
