# -*- encoding: utf-8 -*-
import logging
import memcache

from travel.avia.api_gateway.settings import MEMCACHED_HOST

logger = logging.getLogger(__name__)


class Stat:
    _client = None
    _increment_buffer = []
    hit_key = 'request_{}_hit_ammm'
    time_key = 'request_{}_time_ammm'

    @classmethod
    def client(cls):
        # type: () -> memcache.Client
        if not cls._client:
            cls._client = memcache.Client([MEMCACHED_HOST])

        return cls._client

    @classmethod
    def _get_hit_key(cls, service):
        # type: (str) -> str
        return cls.hit_key.format(service)

    @classmethod
    def _get_time_key(cls, service):
        # type: (str) -> str
        return cls.time_key.format(service)

    @classmethod
    def hit(cls, service, time):
        # type: (str, float) -> None
        logger.debug('Request "{}": {}ms'.format(service, int(time)))
        cls._increment_buffer.append(
            {
                'key': cls._get_hit_key(service),
                'delta': 1,
            }
        )
        cls._increment_buffer.append(
            {
                'key': cls._get_time_key(service),
                'delta': int(time),
            }
        )

    @classmethod
    def flush(cls):
        for data in cls._increment_buffer:
            cls.incr(data['key'], data['delta'])
        cls._increment_buffer = []

    @classmethod
    def incr(cls, key, delta=1):
        # type: (str, int) -> int
        return cls.client().incr(key, delta) or cls.client().add(key, delta)

    @classmethod
    def get_and_clear(cls, services):
        keys = set([cls._get_hit_key(service) for service in services])
        keys.update([cls._get_time_key(service) for service in services])

        result = cls.client().get_multi(keys)

        cls.client().set_multi(dict.fromkeys(keys, 0))

        return result
