from urllib.parse import SplitResult, urlsplit
from os.path import join

from enum import Enum
from dataclasses import dataclass


class AvatarsInstallationType(str, Enum):
    TESTING = 'testing'
    PRODUCTION = 'production'


class Scheme(str, Enum):
    http = 'http'
    https = 'https'


@dataclass
class AvatarsInstallationUrl:
    scheme: str
    netloc: str


@dataclass
class AvatarsInstallationUrls:
    public: AvatarsInstallationUrl
    internal: AvatarsInstallationUrl


AVATARS_URL_FOR_INSTALLATION = {
    AvatarsInstallationType.PRODUCTION: AvatarsInstallationUrls(
        public=AvatarsInstallationUrl(
            netloc='avatars.mds.yandex.net',
            scheme=Scheme.https,
        ),
        internal=AvatarsInstallationUrl(
            netloc='avatars-int.mds.yandex.net:13000',
            scheme=Scheme.http,
        ),
    ),
    AvatarsInstallationType.TESTING: AvatarsInstallationUrls(
        public=AvatarsInstallationUrl(
            netloc='avatars.mdst.yandex.net',
            scheme=Scheme.https,
        ),
        internal=AvatarsInstallationUrl(
            netloc='avatars-int.mdst.yandex.net:13000',
            scheme=Scheme.http,
        ),
    ),
}

_installation_type_by_netloc_ = {
    netloc: installation_type
    for installation_type, urls in AVATARS_URL_FOR_INSTALLATION.items()
    for netloc in [urls.public.netloc, urls.internal.netloc]
}


class Operation(str, Enum):
    get = 'get'
    getimageinfo = 'getimageinfo'


@dataclass
class AvatarsUrlDescriptor:
    installation: AvatarsInstallationUrl
    operation: Operation
    namespace: str
    group_id: str
    image_name: str

    def url(self) -> str:
        return SplitResult(
            scheme=self.installation.scheme,
            netloc=self.installation.netloc,
            path=join(f'{self.operation}-{self.namespace}', self.group_id, self.image_name),
            query=None,
            fragment=None,
        ).geturl()

    def url_for_image_alias(self, alias: str) -> str:
        return SplitResult(
            scheme=self.installation.scheme,
            netloc=self.installation.netloc,
            path=join(f'{self.operation}-{self.namespace}', self.group_id, self.image_name, alias),
            query=None,
            fragment=None,
        ).geturl()

    def url_for_path(self, path: str) -> str:
        return SplitResult(
            scheme=self.installation.scheme,
            netloc=self.installation.netloc,
            path=path,
            query=None,
            fragment=None,
        ).geturl()

    @classmethod
    def from_orig_image_url(cls, original_image_url: str, operation: Operation) -> 'AvatarsUrlDescriptor':
        def installation_for_operation(split: SplitResult, operation: Operation):
            installation_type: AvatarsInstallationType = _installation_type_by_netloc_.get(split.netloc)
            if not installation_type or installation_type not in AVATARS_URL_FOR_INSTALLATION:
                return AvatarsInstallationUrl(
                    netloc=split.netloc,
                    scheme=split.scheme,
                )
            installation_urls: AvatarsInstallationUrls = AVATARS_URL_FOR_INSTALLATION[installation_type]
            return {
                Operation.get: installation_urls.public,
                Operation.getimageinfo: installation_urls.internal,
            }[operation]

        url_split = urlsplit(original_image_url)
        opspace, group_id, image_name, *_ = url_split.path.strip('/').split('/')
        _, namespace = opspace.split('-')
        return AvatarsUrlDescriptor(
            installation=installation_for_operation(url_split, operation),
            operation=operation,
            namespace=namespace,
            group_id=group_id,
            image_name=image_name,
        )
