# coding=utf-8
from jinja2 import Environment

from travel.avia.api_gateway.application.cache.tanker_text_cache import TankerTextCache


def get_point_title_nominative(point, lang='ru'):
    if not point:
        return None

    def extract_title():
        title = point.Title if hasattr(point, 'Title') else point.TitleNominative
        if isinstance(title, str):
            return title
        title = getattr(title, lang.capitalize(), point.TitleDefault)
        if isinstance(title, str):
            return title
        return title.Nominative

    return extract_title() or point.TitleDefault


def _decline(point, declension):
    try:
        return getattr(point.Title.Ru, declension)
    except:
        return get_point_title_nominative(point)


def _get_preposition(point):
    default_preposition = 'в'
    try:
        if point.Title.Ru.LocativePreposition:
            return point.Title.Ru.LocativePreposition
    except:
        pass
    return default_preposition


POINT_DECLENSION_FILTERS = {
    'nominative': lambda x: get_point_title_nominative(x),
    'genitive': lambda x: _decline(x, 'Genitive'),
    'accusative': lambda x: _decline(x, 'Accusative'),
    'prepositional': lambda x: _decline(x, 'Prepositional'),
    'preposition': _get_preposition,
}


class LandingTemplater:
    def __init__(self, tanker_text_cache: TankerTextCache):
        self._tanker_text_cache = tanker_text_cache
        self._env = self._prepare_env()
        self._template_by_key = {}
        for key, value in tanker_text_cache.items():
            self._template_by_key[key] = self._env.from_string(value)

    def _prepare_env(self):
        env = Environment()

        for filter_name in POINT_DECLENSION_FILTERS:
            env.filters[filter_name] = POINT_DECLENSION_FILTERS[filter_name]

        env.filters['capitalize'] = lambda s: s.capitalize()

        return env

    def render(self, key, lang='ru', **kwargs):
        if key not in self._template_by_key:
            return None
        return self._template_by_key[key].render(**kwargs)
