import re
from typing import Union, Optional

from travel.proto.dicts.rasp.settlement_pb2 import TSettlement
from travel.proto.dicts.rasp.station_pb2 import TStation
from travel.proto.dicts.rasp.country_pb2 import TCountry

from travel.avia.library.python.urls.search import S, C, L

SETTLEMENT_POINT_KEY_FORMAT = re.compile(r'^c\d+$')
STATION_POINT_KEY_FORMAT = re.compile(r'^s\d+$')
KRYM_CODE = 'KRYM'
SIMFEROPOL_ID = 146


def get_point_key(point: Union[TStation, TSettlement, TCountry]) -> str:
    if isinstance(point, TStation):
        return str(S(point.Id))
    if isinstance(point, TSettlement):
        return str(C(point.Id))
    if isinstance(point, TCountry):
        return str(L(point.Id))
    raise TypeError(f'Invalid point type {type(point)}')


def get_settlement_id(point: Union[TStation, TSettlement]) -> int:
    if isinstance(point, TStation):
        return point.SettlementId
    if isinstance(point, TSettlement):
        return point.Id
    raise TypeError(f'Invalid point type {type(point)}')


def get_settlement_by_code(cache_root, code) -> Optional[TSettlement]:
    if not code:
        return None
    if SETTLEMENT_POINT_KEY_FORMAT.match(code):
        try:
            settlement_id = int(code[1:])
            return cache_root.settlement_cache.get_settlement_by_id(settlement_id)
        except:
            raise Exception('Invalid settlement point key: "{}"'.format(code))
    elif STATION_POINT_KEY_FORMAT.match(code):
        try:
            station_id = int(code[1:])
            station = cache_root.station_cache.get_station_by_id(station_id)
            if station and station.SettlementId:
                return cache_root.settlement_cache.get_settlement_by_id(station.SettlementId)
        except:
            raise Exception('Invalid station point key: "{}"'.format(code))
    else:
        code_upper = code.upper()
        if code_upper == KRYM_CODE:
            return cache_root.settlement_cache.get_settlement_by_id(SIMFEROPOL_ID)

        settlement = cache_root.settlement_cache.get_settlement_by_slug(code) \
            if cache_root.settlement_cache.get_settlement_by_slug(code) \
            else cache_root.settlement_cache.get_settlement_by_slug(code_upper)
        if settlement:
            return settlement

        station = cache_root.station_cache.get_station_by_slug(code) \
            if cache_root.station_cache.get_station_by_slug(code) \
            else cache_root.station_cache.get_station_by_slug(code_upper)
        if station and station.SettlementId:
            return cache_root.settlement_cache.get_settlement_by_id(station.SettlementId)

        settlement = cache_root.settlement_cache.get_settlement_by_iata(code) \
            if cache_root.settlement_cache.get_settlement_by_iata(code) \
            else cache_root.settlement_cache.get_settlement_by_iata(code_upper)
        if settlement:
            return settlement

        settlement = cache_root.settlement_cache.get_settlement_by_sirena_id(code) \
            if cache_root.settlement_cache.get_settlement_by_sirena_id(code) \
            else cache_root.settlement_cache.get_settlement_by_sirena_id(code_upper)
        if settlement:
            return settlement

        station_id = cache_root.station_code_cache.get_station_id_by_code(code) \
            if cache_root.station_code_cache.get_station_id_by_code(code) \
            else cache_root.station_code_cache.get_station_id_by_code(code_upper)
        station = cache_root.station_cache.get_station_by_id(station_id)
        if station and station.SettlementId:
            return cache_root.settlement_cache.get_settlement_by_id(station.SettlementId)

    return None
