# -*- coding: utf-8 -*-
from collections import OrderedDict
from functools import wraps
from inspect import getargspec

from django.conf import settings
from django.utils.http import urlencode
from flask import url_for, request
from werkzeug.exceptions import BadRequest

from travel.avia.avia_api.ant.argparser import ArgParser
from travel.avia.avia_api.ant.argument import Argument
from travel.avia.avia_api.ant.custom_types import ArgMarschmallowSchema
from travel.avia.avia_api.ant.exceptions import ValidationError, AntException
from travel.avia.avia_api.avia.lib.decorators import skip_None_values
from travel.avia.avia_api.avia.lib.helpers import view


class Ant(object):
    def __init__(
        self, app,
        auth_decorator=None,
    ):
        self.app = app
        self.auth_decorator = auth_decorator
        self._hands = dict()

    def param(self, dst, *args, **kwargs):
        """
        Decorator to add argument from request.values
        """

        def deco(fn):
            self.add_fn_param(fn, dst, *args, **kwargs)

            fn = self.process_viewparams(fn)

            return fn

        return deco

    def add_fn_param(self, fn, dst, *args, **kwargs):
        hand = self.get_or_create_fn_hand(fn)
        hand.update_fn(fn)

        hand.params.insert(
            0,
            self.create_argument_from_fn_param(dst, *args, **kwargs)
        )

    def append_fn_param(self, fn, dst, *args, **kwargs):
        hand = self.get_or_create_fn_hand(fn)
        hand.update_fn(fn)
        hand.params.append(
            self.create_argument_from_fn_param(dst, *args, **kwargs)
        )

    def create_argument_from_fn_param(self, dst, *args, **kwargs):
        kwargs = kwargs.copy()

        kwargs['dst'] = dst

        if 'name' not in kwargs:
            kwargs['name'] = dst

        schema = kwargs.pop('schema', None)

        if schema:
            kwargs['type_'] = ArgMarschmallowSchema(schema)

        return Argument(*args, **kwargs)

    def get_or_create_fn_hand(self, fn):
        """
        Достать или создать обёртку Hand функции fn.
        Можно идентифицировать функцию по имени
        потому что flask не позволяет одноимённые route.
        """
        name = fn.__name__

        if name not in self._hands:
            self._hands[name] = Hand(fn, self)

        return self._hands[name]

    def process_viewparams(self, fn):
        # Явное указание обработать параметры ViewParam и обернуть
        # в функцию без таких параметров.
        # Здесь нужно уже создать hand. И в её fn положить обёрнутую fn

        hand = self.get_or_create_fn_hand(fn)

        if getattr(hand, '_viewparams_processed', False):
            return fn

        # Syntastic sugar to add view param from argument default
        fn_spec = getargspec(fn)
        keyword_arguments = reversed(zip(
            reversed(fn_spec.args),
            reversed(fn_spec.defaults or [])
        ))

        for arg_name, arg_default_value in keyword_arguments:
            if isinstance(arg_default_value, ViewParam):
                viewparam = arg_default_value

                self.append_fn_param(
                    fn, unicode(arg_name),
                    *viewparam.args, **viewparam.kwargs
                )

        # FIXME create function with same signature but without viewparams
        @wraps(fn)
        def processor_viewparams(*args, **kwargs):
            kwargs2 = kwargs.copy()
            kwargs2.update(self.get_params(hand))

            try:
                return fn(*args, **kwargs2)

            except ValidationError as e:
                info = OrderedDict([
                    ('errors', [
                        {'error': (e.fields, str(e))},
                    ]),
                ])

                raise BadRequest(info)

        hand.update_fn(processor_viewparams)
        hand._viewparams_processed = True

        return processor_viewparams

    def result_schema(self, schema_cls, **schema_kwargs):
        """
        Decorator to specify result schema (marshmallow)
        """

        def result_schema_installer(fn):
            """ Wrap result with schema.dump().data """

            fn = self.process_viewparams(fn)

            hand = self.get_or_create_fn_hand(fn)
            hand.result_schema = {'cls': schema_cls, 'kwargs': schema_kwargs}

            @wraps(fn)
            def result_dump_with_schema(*args, **view_kwargs):
                # get_kwargs_spec unused because fn signature changed
                # if 'schema_context' not in view_kwargs and \
                #         'schema_context' in get_kwargs_spec(fn):
                view_kwargs.setdefault('schema_context', {})

                # Actually call wrapped fn
                result = fn(*args, **view_kwargs)

                schema_kwargs_copy = schema_kwargs.copy()

                schema_kwargs_copy.update({
                    'context': view_kwargs['schema_context'],
                    'strict': True,
                })

                return schema_cls(**schema_kwargs_copy).dump(result).data

            hand.update_fn(result_dump_with_schema)

            return result_dump_with_schema

        return result_schema_installer

    def view(self, url, *args, **kwargs):
        """
        Decorator to render flask response
        """

        def deco(fn):
            """ Add flask.route, rendering, viewparams handling """

            fn = self.process_viewparams(fn)

            hand = self.get_or_create_fn_hand(fn)

            hand.update_fn(fn)

            if 'make_spec' in kwargs:
                hand.make_spec = kwargs.pop('make_spec')

            if 'sort_order' in kwargs:
                hand.sort_order = kwargs.pop('sort_order')

            if 'argparser' in kwargs:
                hand.argparser = kwargs.pop('argparser')

            if 'methods' not in kwargs:
                kwargs['methods'] = ['GET', 'POST']

            environments = kwargs.pop('disable_auth_in_environments', [])

            view_deco = view(self.app, url, *args, **kwargs)

            if self.auth_decorator:
                if settings.YANDEX_ENVIRONMENT_TYPE not in environments:
                    fn = self.auth_decorator(fn)

            return view_deco(fn)

        return deco

    @property
    def spec_hand(self):
        spec_fn = getattr(self, '_specification_fn', False)

        if not spec_fn:
            raise AntException('No specification function')

        return self.get_hand(spec_fn)

    @property
    def method_spec_hand(self):
        method_spec_fn = getattr(self, '_method_specification_fn', False)

        if not method_spec_fn:
            raise AntException(
                'No method specification function. '
                'You have to wrap any blank function with '
                '<your_Ant_instance>.specification_view(<route>)'
            )

        return self.get_hand(method_spec_fn)

    @property
    def spec_forms_hand(self):
        forms_fn = getattr(self, '_specification_forms_fn', False)

        if not forms_fn:
            raise AntException('No specification forms function')

        return self.get_hand(forms_fn)

    @property
    def methods(self):
        return OrderedDict([
            (hand.method, hand)
            for hand in sorted(
                self._hands.values(),
                key=lambda h: (h.sort_order is None, h.sort_order)
            )
            if hand.make_spec
        ])

    def get_params(self, hand):
        try:
            return hand.argparser.process(hand.params)

        except ValidationError:
            raise BadRequest(self.hand_erorrs_info(hand))

    def hand_erorrs_info(self, hand):
        info = OrderedDict([
            ('errors', [
                {
                    'error': (arg.name, msg),
                    'help': arg.arg_spec,
                }
                for arg, msg in hand.argparser.args_errors
            ]),
        ])
        return info

    def get_hand(self, key):
        """ Returns hand by fn or fn name or method name """
        try:
            if isinstance(key, Hand):
                return key

            elif callable(key):
                return self._hands[key.__name__]

            elif isinstance(key, basestring):
                return self._hands.get(key) or self.get_hand_of_method(key)

        except Exception:
            pass

        raise AntException('No hand associated with %s', repr(key))

    def get_hand_of_method(self, method_name):
        return self.methods[method_name]

    def get_method_specification_full_url(self, method):
        return self.method_spec_hand.full_url(
            m=self.get_hand(method).method
        )

    @property
    @skip_None_values
    def api_spec(self):
        return OrderedDict([
            (method, hand.hand_spec)
            for method, hand in self.methods.items()
        ])


class Hand(object):

    """
    Ручка api
    Здесь хранится описание аргументов
    """

    def __init__(self, fn, ant):
        self.fn = fn
        self.ant = ant
        self.params = list()
        self.make_spec = True
        self.sort_order = None
        self.argparser = ArgParser()

    def update_fn(self, fn):
        self.fn = fn

    @property
    @skip_None_values
    def hand_spec(self):
        return {
            'about': self.about,
            'params': OrderedDict(
                (p.name, p.arg_spec) for p in self.params
            ) or None,
            'address': self.full_url(),
        }

    @property
    def hand_form(self):
        return {
            'about': self.about or self.method,
            'address': self.full_url(),
            'form': [
                skip_None_values({
                    'key': p.name,
                    'placeholder': p.type_.type_spec.get('format'),
                    'type': p.type_.form_type,
                }) for p in self.params
            ],
            'schema': skip_None_values({
                'type': 'object',
                'title': self.about or self.method,
                'properties': OrderedDict([
                    (p.name, p.arg_form_schema) for p in self.params
                ]),
                'required': [
                    p.name for p in self.params if p.required
                ] or None,
            }),
        }

    @property
    def about(self):
        return self.fn.__doc__ and self.fn.__doc__.strip() or None

    @property
    def method(self):
        return self.url()

    def url(self, **kwargs):
        params = OrderedDict()

        for param in self.params:
            if param in kwargs:
                params[param] = kwargs.pop(param)

        url = url_for('.' + self.fn.__name__, **kwargs)
        if params:
            url += '?' + urlencode(params)

        return url

    def full_url(self, **kwargs):
        return (
            'https://' +
            request.environ['HTTP_HOST'] +
            self.url(**kwargs)
        )

    def spec_method_full_url(self):
        return self.ant.get_method_specification_full_url(self)

    def __str__(self):
        return self.method

    def __repr__(self):
        return '<%s %s>' % (self.__class__.__name__, self.method)


class ViewParam(object):

    """
    Object to pass as default value to @view-decorated function.
    Then Argument will be created for this function specification.
    """

    def __init__(self, *args, **kwargs):
        self.args = args
        self.kwargs = kwargs
