# -*- coding: utf-8 -*-
import logging

from flask import request

from travel.avia.avia_api.ant.argument import Argument
from travel.avia.avia_api.ant.exceptions import ValidationError
from travel.avia.avia_api.avia.lib.remote_addr import real_client_ip

log = logging.getLogger(__name__)


class ArgParser(object):
    def process(self, arguments):
        raw_args = {
            arg: self.get_arg_raw_value(arg)
            for arg in arguments
        }

        self.args_errors = []

        cleaned = self.clean_args(raw_args)

        if self.args_errors:
            raise ValidationError('Arguments process error')

        for arg in arguments:
            if arg.afterclean:
                cleaned[arg.dst] = arg.afterclean(cleaned[arg.dst])

        return cleaned

    def clean_args(self, raw_args):
        return dict(self._clean_args(raw_args))

    def _clean_args(self, raw_args):
        for arg, raw in raw_args.items():
            try:
                yield (arg.dst, self.clean_arg_value(arg, raw))

            except Exception:
                pass

    def clean_arg_value(self, arg, raw):
        try:
            return arg.clean(raw)

        except ValidationError as e:
            try:
                msg = str(e)
            except UnicodeEncodeError:
                msg = repr(e.message)
            self.args_errors.append((arg, msg))

            raise

    def get_arg_raw_value(self, arg):
        assert isinstance(arg, Argument)

        if request.method == 'GET':
            return request.values.get(arg.name)

        if request.method == 'POST':
            if request.values:
                return request.values.get(arg.name)

            data = request.get_json(force=True, silent=True)
            if data is not None:
                return data.get(arg.name)


class LoggingRawArgParser(ArgParser):
    """ Парсер аргументов вьюшки, логирующий неразобранные значения """

    def __init__(self, log):
        super(LoggingRawArgParser, self).__init__()
        self.log = log

    def clean_args(self, raw_args):
        try:
            return (
                super(LoggingRawArgParser, self).clean_args(raw_args)
            )

        finally:
            try:
                self.log_raw_args(raw_args)
                request.environ['raw_args'] = raw_args

            except Exception:
                log.exception('Error logging raw_args')

    def log_raw_args(self, raw_args):
        args = {a.name: v for a, v in raw_args.items()}

        user_info = 'ip:%s' % real_client_ip(request)

        if self.args_errors:
            errors = {a.name: e for a, e in self.args_errors}
            self.log.warning('%s %r Bad: %r', user_info, args, errors)

        else:
            self.log.info('%s %r', user_info, args)
