from collections import OrderedDict

import logging

from travel.avia.avia_api.ant.custom_types import ArgType, Str
from travel.avia.avia_api.ant.exceptions import ValidationError, AntException
from travel.avia.avia_api.avia.lib.decorators import (
    pipe_out, skip_None_values, refuse_None, elementwise
)

log = logging.getLogger(__name__)


@elementwise
def process_choice(raw):
    return raw if isinstance(raw, tuple) else (raw, str(raw))


@refuse_None
@pipe_out(OrderedDict)
def process_choices(raw_choices):
    if callable(raw_choices):
        return process_choices(raw_choices())

    elif isinstance(raw_choices, dict):
        return process_choice(raw_choices.items())

    # Assume raw_choices is iterable
    return process_choice(raw_choices)


class Argument(object):

    """
    Stores arguments for ArgParser.process
    """

    def __init__(self, name, dst=None, default=None, type_=None, choices=None,
                 required=True, about=None, afterclean=None):
        if type_ and not isinstance(type_, ArgType):
            raise AntException('type_ must be an instance of ArgType subclass.'
                               ' Got %s' % repr(type_))

        self.name = name
        self.dst = dst or name
        self.default = default
        self.type_ = type_ or Str()
        self._choices = choices
        self.required = required and default is None
        self._about = about
        self._afterclean = afterclean

    @property
    def afterclean(self):
        return self._afterclean or \
            getattr(self.type_, 'argtype_afterclean', None)

    @property
    def about(self):
        return (self._about or self.type_.__doc__ or '').strip() or None

    @property
    @pipe_out(process_choices)
    def choices(self):
        if not self._choices:
            # Use choices from argument type if available
            return getattr(self.type_, 'choices', None)

        if callable(self._choices):
            return self._choices()

        return self._choices

    def clean(self, raw):
        if raw is None:
            if self.required:
                raise ValidationError('Required parameter missed')
            return self.default

        try:
            val = self.type_.clean(raw)

        except ValidationError:
            raise

        except NotImplementedError:
            raise

        except Exception:
            log.exception('Argument %s validation error', repr(self.type_))

            raise ValidationError('Bad value')

        if self.choices and val not in self.choices:
            raise ValidationError('Value not in choices')

        return val

    @property
    @skip_None_values
    def arg_spec(self):
        spec = self.type_.type_spec.copy()

        spec.update(OrderedDict([
            ('about', self.about),
            ('required', self.required),
            ('default', self.default),
            ('choices', self.choices_spec),
        ]))

        return spec

    @property
    def choices_spec(self):
        return self.choices and \
            OrderedDict([(str(key), val) for key, val in self.choices.items()])

    @property
    @skip_None_values
    def arg_form_schema(self):
        schema = self.type_.type_schema.copy()

        enum = map(str, OrderedDict(self.choices or []).keys())

        schema.update(skip_None_values({
            'title': self.about,
            'enum': enum or None,
            'default': self.default or enum[0] if enum else None,
        }))

        return schema

    def __repr__(self):
        return '<%s %s>' % (self.__class__.__name__, self.name)
