# -*- coding: utf-8 -*-
import json
import logging
from collections import OrderedDict
from datetime import datetime

import pytz
import re
from django.conf import settings
from django.utils.functional import cached_property
from flask import request

from travel.avia.library.python.common.models.currency import Currency
from travel.avia.library.python.common.models.geo import Station, Settlement
from travel.avia.library.python.common.models_utils.geo import Point

from travel.avia.avia_api.ant.exceptions import ValidationError
from travel.avia.avia_api.avia.api.daemon_api import DaemonQuery, HttpChance
from travel.avia.avia_api.avia.cache.settlements import avia_settlement_cache, settlement_cache
from travel.avia.avia_api.avia.cache.stations import airport_cache, station_cache
from travel.avia.avia_api.avia.lib.app_user import AppUser
from travel.avia.avia_api.avia.lib.decorators import skip_None_values
from travel.avia.avia_api.avia.lib.flight_classes import FlightClass
from travel.avia.avia_api.avia.lib.serialization import (
    CsvCryptable, str_converter, asis_converter,
    date_converter, datetime_converter,
)

log = logging.getLogger(__name__)


class ArgType(object):
    def __init__(self, *args, **kwargs):
        pass

    def clean(self, raw):
        return raw

    @property
    def type_spec(self):
        return {'format': self.fmt}

    fmt = None

    @property
    @skip_None_values
    def type_schema(self):
        return {
            'type': self.schema_type,
        }

    schema_type = 'string'
    form_type = None


class ArgLang(ArgType):
    choices = settings.AVIA_API_LANGUAGES


class HttpSearchIdentificator(CsvCryptable):
    _serialize_attrs = OrderedDict([
        ('qkey', str_converter),
        ('timestamp', datetime_converter),

        ('_qid', str_converter),
        ('_national_version', str_converter),
        ('_point_from_key', str_converter),
        ('_point_to_key', str_converter),
        ('_date_forward', date_converter),
    ])

    def __init__(
        self, qid, national_version, point_from_key, point_to_key, date_forward
    ):
        self.qkey = ''  # legacy
        self._qid = qid
        self._national_version = national_version
        self._point_from_key = point_from_key
        self._point_to_key = point_to_key
        self._date_forward = date_forward
        self.timestamp = request.environ['now']

    def qid(self):
        return self._qid

    def national_version(self):
        return self._national_version

    def point_from(self):
        return Point.get_by_key(self._point_from_key)

    def point_to(self):
        return Point.get_by_key(self._point_to_key)

    def date_forward(self):
        return self._date_forward


class SovetnikSearchIdentificator(CsvCryptable):
    _serialize_attrs = OrderedDict([
        ('_qkey', str_converter),
        ('_currency', str_converter),
        ('_lang', str_converter),
        ('_clid', str_converter),
        ('_skip', str_converter),
        ('_variant_count', asis_converter),
        ('_qid', str_converter),
        ('_nv', str_converter),
    ])

    def __init__(self, daemon_query, qid, currency, lang, clid,
                 national_version, partners_to_skip=tuple(),
                 skipped_variant_count=0):

        self._query = daemon_query
        self._qid = qid
        self._currency = currency
        self._lang = lang
        self._clid = clid
        self._partners_to_skip = tuple(partners_to_skip)
        self._variant_count = skipped_variant_count
        self._nv = national_version

    def query(self):
        return self._query

    def qid(self):
        return self._qid

    def partners_to_skip(self):
        return list(self._partners_to_skip)

    def currency(self):
        return self._currency

    def language(self):
        return self._lang

    def clid(self):
        return self._clid

    def national_version(self):
        try:
            return self._nv
        except AttributeError:
            # Нужно удалить эту проверку после выкатки в стейбл
            log.error('No _nv on SovetnikSearchIdentificator')
            return 'ru'

    def skipped_variant_count(self):
        try:
            return self._variant_count
        except AttributeError:
            # Нужно удалить эту проверку после выкатки в стейбл
            log.error('No _variant_count on SovetnikSearchIdentificator')
            return 0

    def with_also_skipped(self, partner_codes, variant_count):
        already_skipped = set(self._partners_to_skip)
        if any(code in already_skipped for code in partner_codes):
            log.error(
                'Partners are already skipped in %s. (%s, %s)',
                self, already_skipped, partner_codes
            )

        return SovetnikSearchIdentificator(
            daemon_query=self._query,
            qid=self.qid(),
            partners_to_skip=tuple(sorted(set(
                self._partners_to_skip + tuple(partner_codes)
            ))),
            currency=self._currency,
            lang=self._lang,
            clid=self._clid,
            national_version=self.national_version(),
            skipped_variant_count=self.skipped_variant_count() + variant_count,
        )

    @property
    def _qkey(self):
        return self._query.key()

    @_qkey.setter
    def _qkey(self, qkey):
        self._query = DaemonQuery.from_key(qkey)

    @property
    def _skip(self):
        # CsvCryptable не умеет работать со строками с запятыми, поэтому нельзя
        # просто сделать json.dumps для массива длиной больше 1
        return ';'.join(self._partners_to_skip)

    @_skip.setter
    def _skip(self, data):
        self._partners_to_skip = tuple(data.split(';') if data else [])


class ArgHttpSearchId(ArgType):
    def clean(self, raw):
        try:
            return HttpSearchIdentificator.create_from_crypted(raw)
        except Exception as e:
            log.error('HttpSearchIdentificator clean: %r %r', e, raw)
            raise ValidationError('Unknown http search identificator')


class ArgSovetnikSearchId(ArgType):
    def clean(self, raw):
        try:
            return SovetnikSearchIdentificator.create_from_crypted(raw)
        except Exception as e:
            log.error('SovetnikSearchIdentificator clean: %r %r', e, raw)
            raise ValidationError('Unknown sovetnik search identificator')


class ArgTags(ArgType):
    """ Идентификаторы вариантов vtag через запятую """

    @property
    def fmt(self):
        return u'Vtag из ответа ручки search_results'

    def clean(self, raw):
        return raw.split(',')


class ArgHttpChance(ArgType):
    def clean(self, raw):
        return HttpChance.create_from_crypted(raw)


def point_by_any_key(raw):
    try:
        return Point.get_by_key(raw)
    except Exception:
        pass

    raw_low = raw.lower()

    try:
        return Settlement.objects.get(iata__iexact=raw_low)
    except Exception:
        pass

    try:
        return Settlement.objects.get(sirena_id__iexact=raw_low)
    except Exception:
        pass

    try:
        return (
            Station.objects
            .filter(code_set__system__code='iata')
            .distinct()
            .get(code_set__code__iexact=raw_low)
        )
    except Exception:
        pass

    try:
        return (
            Station.objects
            .filter(code_set__system__code='sirena')
            .distinct()
            .get(code_set__code__iexact=raw_low)
        )
    except Exception:
        pass

    return None


class ArgPoint(ArgType):
    fmt = 'Внутренний код объекта из справочника, iata—код или сирена—код'

    def clean(self, raw):
        point = point_by_any_key(raw)
        if point:
            return point

        raise ValidationError('Point not found %r' % raw)


def prefixes(string):
    parts = string.split()
    return (
        ' '.join(parts[:idx]) for idx in range(1, len(parts))
    )


def sovetnik_point_by_title(raw):
    raw = raw.strip()
    if not raw:
        return None

    def _try(title):
        return (
            avia_settlement_cache.by_title(title) or
            airport_cache.by_title(title) or
            settlement_cache.by_title(title) or
            station_cache.by_title(title) or
            [None]
        )[0]

    point = (
        point_by_any_key(raw) or
        _try(raw) or
        _try(raw.split(',')[0]) or
        _try(raw.split('.')[0]) or
        _try(raw.split('(')[0]) or
        _try(raw.split('-')[0])
    )

    for prefix in sorted(prefixes(raw), key=len, reverse=True):
        point = point or _try(prefix)

    point = point or _try(re.split(r'[\W\d]+', raw, re.UNICODE)[0])

    if point:
        log.info('Sovetnik point: %r -> %s', raw, point.point_key)
    else:
        log.info('Sovetnik point is not found for %r', raw)

    return point


class ArgSovetnikPoint(ArgType):
    def clean(self, raw):
        point = sovetnik_point_by_title(raw)
        if point:
            return point
        raise ValidationError('Point for %r is not found' % raw)


class ArgSovetnikDynamicsSettlement(ArgType):
    def clean(self, raw):
        point = sovetnik_point_by_title(raw)

        if point:
            if isinstance(point, Station):
                cities = avia_settlement_cache.by_station_id(point.id)
                if cities:
                    point = cities[0]

            if isinstance(point, Settlement):
                return point

        raise ValidationError('Settlement for %r is not found' % raw)


class Month(object):
    def __init__(self, year, month):
        self._year = year
        self._month = month

    @classmethod
    def from_datetime(cls, dt):
        return cls(dt.year, dt.month)

    def year(self):
        return self._year

    def month(self):
        return self._month

    def first_day(self):
        return datetime(self._year, self._month, 1)

    def plus(self, month_count):
        years, new_month = divmod(self._month + month_count - 1, 12)
        return Month(self._year + years, new_month + 1)


class ArgMonth(ArgType):
    """
    Example: "2017-12"
    """
    def clean(self, raw):
        try:
            return Month.from_datetime(datetime.strptime(raw, '%Y-%m'))
        except Exception:
            raise ValidationError('Month %r is not valid' % raw)


class ArgSearchPages(ArgType):
    fmt = 'json'

    def clean(self, raw):
        try:
            pages = raw and json.loads(raw) or {}
            assert isinstance(pages, dict)
            pages = {code: int(page) for code, page in pages.items()}

        except ValidationError:
            raise

        except Exception:
            log.exception('ArgSearchPages')
            raise ValidationError('Bad pages %r' % raw)

        return pages


class ArgCSPartnersCodes(ArgType):
    """ Json кодов партнёров """
    fmt = 'json'

    def clean(self, raw):
        if not raw:
            return []
        try:
            return filter(None, json.loads(raw))
        except Exception as exc:
            log.exception('ArgCSPartnersCodes: %r', exc)
            raise ValidationError('Bad codes %r' % raw)


class ArgAppUserByAppKey(ArgType):
    """ Ключ приложения """

    def clean(self, raw):
        return AppUser.get_by_app_key(raw)


class ArgUUID(ArgType):
    """ Валидный UUID """

    def clean(self, raw):
        if not re.match(ur'^[a-f0-9]{32}$', raw):
            raise ValidationError('Invalid UUID: %r' % raw)
        return raw


class ArgPassportId(ArgType):
    """ Passport ID пользователя """

    def clean(self, raw):
        # type: (str) -> str
        if not raw.isdigit():
            raise ValidationError('Invalid passport id: %r' % raw)
        return raw


class ArgsList(ArgType):
    def __init__(self, arg_type):
        super(ArgsList, self).__init__()
        self.arg_type_instance = arg_type

    def clean(self, raw):
        if isinstance(raw, basestring):
            try:
                raw = json.loads(raw)
            except Exception:
                pass

        if not isinstance(raw, list):
            raise ValidationError('Strange list of {!r}: {!r}'.format(
                self.arg_type_instance, raw
            ))

        return [
            self.arg_type_instance.clean(value)
            for value in raw
        ]


class SettlementGeoIdToStationIdList(ArgType):
    # Tries to convert settlement geo id to list of station ids for each element in the list,
    # passes the element's value as is, if unable to perform a conversion
    def clean(self, raw):
        result = []
        for value in raw.split(' '):
            try:
                result.extend([str(geo_id) for geo_id in airport_cache.by_settlement_geo_id(int(unicode(value)))])
            except:
                result.append(unicode(value))
        return result


class Str(ArgType):
    def clean(self, raw):
        return unicode(raw)


class Bool(ArgType):
    def clean(self, raw):
        return raw.lower() == 'true'


class Int(ArgType):
    def __init__(self, interval=(None, None), *args, **kwargs):
        self.min, self.max = interval

    def clean(self, raw):
        val = int(raw)

        if self.max and val > self.max:
            raise ValidationError('Out of range')

        if self.min and val < self.min:
            raise ValidationError('Out of range')

        return val

    @property
    @skip_None_values
    def type_spec(self):
        return OrderedDict({
            ('min', self.min),
            ('max', self.max),
        })

    @property
    @skip_None_values
    def type_schema(self):
        return OrderedDict({
            ('type', 'integer'),
            ('minimum', self.min),
            ('maximum', self.max),
            ('default', self.min if self.min is not None else self.max),
        })


class ArgCurrency(ArgType):
    def clean(self, raw):
        try:
            return Currency.objects.get(code=raw)
        except Currency.DoesNotExist:
            raise ValidationError(
                'Bad currency. Supported codes: {}'.format(self._available_codes)
            )

    @cached_property
    def _available_codes(self):
        return sorted(Currency.objects.values_list('code', flat=True))


DATE_FMT = '%Y-%m-%d'


class Date(ArgType):
    fmt = 'YYYY-MM-DD'

    def clean(self, raw):
        return datetime.strptime(raw, DATE_FMT).date()


class DateStr(ArgType):
    def clean(self, raw):
        try:
            return unicode(raw)
        except:
            raise ValidationError('Unable to parse {!r} as date'.format(raw))


DATETIME_FMT = '%Y-%m-%d %H:%M:%S'


class DateTime(ArgType):
    fmt = 'YYYY-MM-DD hh:mm:ss'

    def clean(self, raw):
        return datetime.strptime(raw, DATETIME_FMT)


class DateTimeGmt(ArgType):
    fmt = 'YYYY-MM-DD hh:mm:ss (в GMT)'

    def clean(self, raw):
        if raw.endswith('Z'):
            log.warning('Bad GMT datetime: %r', raw)
            raw = raw.rstrip('Z').replace('T', ' ')

        return datetime.strptime(raw, DATETIME_FMT).replace(tzinfo=pytz.utc)


class ArgAntMethod(ArgType):

    """ Метод API """

    def __init__(self, ant):
        self.ant = ant

    @property
    def choices(self):
        return [
            (hand, hand.about)
            for hand in self.ant.methods.values()
        ]

    def clean(self, raw):
        return self.ant.get_hand_of_method(raw)


class ArgFlightClass(ArgType):
    def clean(self, raw):
        try:
            return {
                'A': FlightClass.Economy,
                'B': FlightClass.Business
            }[raw]
        except KeyError:
            raise ValidationError('Wrong flight class: %r', raw)


class ArgMarschmallowSchema(ArgType):
    def __init__(self, schema, *args, **kwargs):
        super(ArgMarschmallowSchema, self).__init__()

        if not args and not kwargs and isinstance(schema, tuple) and len(schema) == 2:
            schema_cls, kwargs = schema
        else:
            schema_cls = schema

        self.schema_cls = schema_cls
        self.schema_kwargs = kwargs

    def get_schema(self):
        return self.schema_cls(**self.schema_kwargs)

    def clean(self, raw):
        data = self.get_schema().opts.json_module.loads(raw)

        errors = self.get_schema().validate(data)

        if errors:
            raise ValidationError(errors)

        return data

    def argtype_afterclean(self, data):
        return self.get_schema().load(data).data
