# coding=utf-8
import logging

import pycountry

from travel.avia.avia_api.ant.custom_types import ArgType
from travel.avia.avia_api.ant.exceptions import ValidationError

log = logging.getLogger(__name__)


class IsoCurrencyArg(ArgType):
    def clean(self, raw):
        try:
            raw = raw.strip().upper()
            if raw == 'RUR':
                return 'RUR'  # Используем RUR вместо RUB
            if raw == 'RUB':
                raise ValidationError('Currency RUB(not RUR) is not correct')
            return pycountry.currencies.get(alpha_3=raw).alpha_3
        except Exception as e:
            log.warning('Strange currency "%r": %s', raw, e)
            raise ValidationError('Currency %r is not correct' % raw)


class IsoLanguageArg(ArgType):
    def clean(self, raw):
        if raw == 'iw':  # hebrew in the old ISO
            raw = 'he'
        try:
            return pycountry.languages.get(alpha_2=raw).alpha_2
        except Exception as e:
            log.warning('Strange language "%r": %s', raw, e)
            raise ValidationError('Language %r is not correct' % raw)


class IsoCountryArg(ArgType):
    def clean(self, raw):
        try:
            return pycountry.countries.get(alpha_2=raw).alpha_2
        except Exception as e:
            log.warning('Strange country "%r": %s', raw, e)
            # RASPTICKETS-6323 - андроид присылает кривые страны, нельзя кидать
            # исключение
            return None


class IsoLocale(object):
    """
    Как такового, ISO по локалям не существует. Но мы будем принимать локаль в
    виде {IsoLang}_{IsoCountry}, потому что мобильные разработчики говорят, что
    так могут, а больше нам и не надо.
    """
    def __init__(self, language, country):
        self.language = language
        self.country = country

    def __unicode__(self):
        return u'''IsoLocale('{}', '{}')'''.format(self.language, self.country)

    def __repr__(self):
        try:
            return str(unicode(self))
        except UnicodeDecodeError:
            return repr(unicode(self))


class IsoLocaleArg(ArgType):
    def clean(self, raw):
        return IsoLocale(
            language=self._lang(raw),
            country=self._country(raw),
        )

    @staticmethod
    def _lang(raw):
        if '_' in raw:
            lang = raw.split('_')[0]
        elif len(raw) == 2 and raw == raw.lower():
            lang = raw
        else:
            return None

        if lang == 'iw':
            lang = 'he'  # hebrew in the old ISO

        try:
            return pycountry.languages.get(alpha_2=lang).alpha_2
        except Exception as e:
            log.warning('Strange lang in locale "%r": %s', raw, e)
            return None

    @staticmethod
    def _country(raw):
        if '_' in raw:
            country = raw.split('_')[1]
        elif len(raw) == 2 and raw == raw.upper():
            country = raw
        else:
            return None

        try:
            return pycountry.countries.get(alpha_2=country).alpha_2
        except Exception as e:
            log.warning('Strange country in locale "%r": %s', raw, e)
            return None
