# -*- coding: utf-8 -*-
import logging
from collections import OrderedDict

from travel.avia.library.python.common.models.geo import Station2Settlement, StationType, Station

from travel.avia.avia_api.avia.api.query import Query
from travel.avia.avia_api.avia.api.redirect import url_add_openstat_marker, get_redirect_partner
from travel.avia.avia_api.avia.lib.serialization import (
    JsonCryptable, asis_converter, str_converter, date_converter,
)

log = logging.getLogger(__name__)


class DaemonQuery(object):
    def __init__(self, query):
        self._origin = query

    @classmethod
    def from_search_args(cls, **kwargs):
        return cls(Query(**kwargs))

    @classmethod
    def from_key(cls, key):
        return cls(Query.from_key(key))

    def key(self):
        return self._origin.key()

    @property
    def point_from(self):
        return self._origin.point_from

    @property
    def point_to(self):
        return self._origin.point_to

    @property
    def date_forward(self):
        return self._origin.date_forward

    @property
    def date_backward(self):
        return self._origin.date_backward

    @property
    def national_version(self):
        return self._origin.national_version

    @property
    def passengers(self):
        return self._origin.passengers

    def search_params(self):
        return {
            'passengers': self.passengers,
            'date_forward': self.date_forward,
            'date_backward': self.date_backward,
            'direction': {
                'arrival': PointReference(self.point_to),
                'departure': PointReference(self.point_from),
            },
            'national_version': self.national_version,
        }

    def __repr__(self):
        return '<DaemonQuery: {!r}>'.format(self._origin)


class DaemonResult(object):
    def __init__(self, variants, statuses):
        self._statuses = statuses
        self._variants = variants

    def available_variants(self, aware_now=None):
        return tuple(
            variant
            for variant in self._variants
            if aware_now is None or variant.booking_available(aware_now)
        )

    def statuses(self):
        return self._statuses.copy()


class HttpChance(JsonCryptable):
    # must contain all fields of a regular chance in order to be logged in
    # the popular_flight_log
    _json_attrs = OrderedDict([
        ('order_data', asis_converter),
        ('forward_numbers', str_converter),
        ('backward_numbers', str_converter),
        ('show_id', str_converter),
        ('national_version', str_converter),
        ('lang', str_converter),
        ('_qid', str_converter),
        ('from_point_key', str_converter),
        ('to_point_key', str_converter),
        ('date_forward', date_converter),
        ('date_backward', date_converter),
    ])

    def __init__(
        self, order_data, show_id, forward_numbers, backward_numbers,
        national_version, lang, qid,
        from_point_key, to_point_key, date_forward, date_backward,
    ):

        self.order_data = order_data
        self.show_id = show_id
        self.forward_numbers = forward_numbers
        self.backward_numbers = backward_numbers
        self.national_version = national_version
        self.lang = lang
        self._qid = qid
        self.from_point_key = from_point_key
        self.to_point_key = to_point_key
        self.date_forward = date_forward
        self.date_backward = date_backward

    def qid(self):
        return self._qid

    def partner(self):
        return get_redirect_partner(self.order_data)

    @classmethod
    def from_variant(cls, variant, national_version, lang, qid):
        order_data = variant.order_data.copy()
        order_data['partner'] = variant.partner.code

        departure_dt = variant.forward.starts_at()
        return_dt = variant.backward.starts_at()

        return cls(
            order_data=order_data,
            forward_numbers=u';'.join(
                segment.number for segment in variant.forward.segments
            ),
            backward_numbers=u';'.join(
                segment.number for segment in variant.backward.segments
            ),
            show_id=variant.show_id,
            national_version=national_version,
            lang=lang,
            qid=qid,
            from_point_key=variant.forward.start_station().point_key,
            to_point_key=variant.forward.end_station().point_key,
            date_forward=departure_dt.date(),
            date_backward=return_dt and return_dt.date(),
        )


class RedirectInfo(object):
    def __init__(self, order_data, redirect_data, partner):
        self._order_data = order_data
        self._redirect_data = redirect_data
        self._partner = partner

    def url(self, http_host):
        url = self._redirect_data.m_url or self._redirect_data.url
        if not url:
            return None

        return url_add_openstat_marker(
            url=url,
            partner_code=self._partner.code,
            host=http_host,
            domain=self._order_data.get('national') or None
        )

    @property
    def post_data(self):
        return self._redirect_data.post

    @property
    def partner(self):
        return self._partner


class PointReference(object):
    """Assumes that the point is either Settlement or Station"""
    def __init__(self, point):
        self._point = point

    def settlement(self):
        if not self._point.is_station:
            return self._point

        try:
            return (
                self._point.settlement or
                Station2Settlement.objects.get(station=self._point).settlement
            )
        except Station2Settlement.DoesNotExist:
            pass

        return None

    def station(self):
        if self._point.is_station:
            return self._point

        try:
            return self._point.station_set.get(
                station_type_id=StationType.AIRPORT_ID
            )
        except Station.MultipleObjectsReturned:
            pass
        except Station.DoesNotExist:
            log.warning('Settlement without an airport: %r', self._point, exc_info=True)

        return None

    def point(self):
        return self._point
