# -*- coding: utf-8 -*-
import copy
import hashlib
import logging
import re
from datetime import timedelta

from django.conf import settings

from travel.avia.library.python.avia_data.models import AmadeusMerchant, AviaCompany, CompanyTariff
from travel.avia.library.python.common.models.geo import Station
from travel.avia.library.python.common.models.partner import Partner, DohopVendor
from travel.avia.library.python.common.models.schedule import Company
from travel.avia.library.python.common.models.tariffs import CLS_NAMES
from travel.avia.library.python.common.models.transport import TransportType
from travel.avia.library.python.common.utils.caching import cache_method_result
from travel.avia.library.python.common.utils.date import MSK_TZ, UTC_TZ
from travel.avia.library.python.common.utils.iterrecipes import pairwise
from travel.avia.library.python.common.views.tariffs import DisplayInfo
from travel.avia.library.python.route_search import transfers

from travel.avia.avia_api.avia.lib.currency import Price
from travel.avia.avia_api.avia.lib.serialization import (
    JsonSerializable, ModelConverter, ToJsonFromJsonConverter,
    IfExistsConverter, ListConverter, DictConverter,
    datetime_converter, asis_converter, datetimeaware_converter,
)

log = logging.getLogger(__name__)
flight_board_log = logging.getLogger('flight_board')

SERVICE_CLASS_PRIORITIES = {'economy': 1, 'business': 2, 'first': 3}
DT_FMT = '%Y-%m-%dT%H:%M:%S'


class BadVariantException(Exception):
    pass


tariff_converter = ToJsonFromJsonConverter(
    lambda p: p and {
        'value': p.value,
        'currency': p.currency,
    } if isinstance(p, Price) else p,
    lambda p: p and
    Price(value=p.get('value'), currency=p.get('currency'))
    if isinstance(p, dict) else p
)


class Flight(JsonSerializable):
    rtstation_from = None
    rtstation_to = None
    thread = None
    baggage = None
    baggage_norm = None
    avia_company = None
    code_tariff = None
    company_tariff = None

    display_t_code = 'plane'

    t_model = None

    supplier_code = None
    charter = None

    gone = False

    _json_attrs = {
        'display_t_code': asis_converter,
        'supplier_code': asis_converter,
        'gone': asis_converter,
        'number': asis_converter,
        'tariff': tariff_converter,  # TODO удалить. Всегда None, т.к. оно хранится в варианте
        'station_from': ModelConverter(Station),
        'station_to': ModelConverter(Station),
        'departure': datetimeaware_converter,
        'arrival': datetimeaware_converter,
        'company': ModelConverter(Company),
    }

    @classmethod
    def _get_subclasses(cls):
        return [
            IATAFlight, UkrmintransTrainRoute, UkrmintransBusRoute,
            BusComUaRoute, SwdfactoryRoute, BlablacarRoute,
        ]

    def __init__(self, **kwargs):
        self.display_info = DisplayInfo()
        self.station_from = None
        self.station_to = None
        self.departure = None
        self.arrival = None
        self.company = None
        self.klass = None
        self.route = None
        self.thread = None

    def link(self, point_from, point_to):
        return None

    def preprocess(self):
        pass

    @property
    def t_type(self):
        if not hasattr(self, '_t_type'):
            self._t_type = TransportType.objects.get(code=self.display_t_code)

        return self._t_type

    def get_title_dict(self):
        return {
            'type': 'default',
            'title_parts': ['from', 'to']
        }

    @property
    def title(self):
        points = self.points_for_title()
        return "%s - %s" % (
            points['from'].L_title(),
            points['to'].L_title(),
        )

    def points_for_title(self):
        station_from = self.station_from
        station_to = self.station_to

        return {
            'from': station_from.settlement or station_from,
            'to': station_to.settlement or station_to,
        }

    def L_title(self, lang=None, short=False, popular=False):
        from travel.avia.library.python.common.utils.title_generator import TitleGeneratorError, TitleGenerator

        if not self.station_from or not self.station_to:
            return 'raw'

        title_dict = self.get_title_dict()
        points = self.points_for_title()

        try:
            return TitleGenerator.L_title(
                title_dict, points,
                short=short, lang=lang, popular=popular
            )
        except TitleGeneratorError:
            return 'raw'

    def get_popular_title(self, lang=None):
        return self.L_title(popular=True, lang=lang)

    @property
    def ticket_number(self):
        return self.route.ticket_number if self.route else self.number

    @property
    def msk_departure(self):
        return self.departure and self.departure.astimezone(MSK_TZ).replace(tzinfo=None)

    @property
    def msk_arrival(self):
        return self.arrival and self.arrival.astimezone(MSK_TZ).replace(tzinfo=None)

    @property
    def utc_departure(self):
        return self.departure and self.departure.astimezone(UTC_TZ).replace(tzinfo=None)

    @property
    def utc_arrival(self):
        return self.arrival and self.arrival.astimezone(UTC_TZ).replace(tzinfo=None)

    @property
    def duration(self):
        return self.arrival and self.departure and self.arrival - self.departure

    @classmethod
    def _cache_key_func(cls, json_data):
        # Генерируем сложный ключ чтобы не потерять данные,
        # если они есть у одного партнёра но нет у другого
        company = json_data.get('company')
        number = json_data.get('number')
        # просто departure&arrival пихать в ключ необязательно
        # т.к. они получаются из того же места что local_
        local_departure = json_data.get('local_departure')
        local_arrival = json_data.get('local_arrival')
        station_from = json_data.get('station_from')
        station_to = json_data.get('station_to')

        # Обязательные параметры без которых кэширование не имеет смысла
        if not company or not number:
            return None

        try:
            return "%s_%s_%s_%s_%s_%s" % (
                number, company.get('id'), local_departure, local_arrival,
                station_from.get('id') if station_from else 'None',
                station_to.get('id') if station_to else 'None')

        except Exception:
            return None

    @cache_method_result
    def cache_key(self):
        # Обязательные параметры без которых кэширование не имеет смысла
        if not self.company or not self.number:
            return None

        return "%s_%s_%s_%s_%s_%s" % (
            self.number, self.company.id,
            self.local_departure.strftime('%Y-%m-%d %H:%M:%S'),
            self.local_arrival.strftime('%Y-%m-%d %H:%M:%S'),
            self.station_from.id if self.station_from else 'None',
            self.station_to.id if self.station_to else 'None')

    def __repr__(self):
        return "<%s %r-%r (%r-%r) [%s]>" % (
            self.__class__.__name__,
            self.station_from, self.station_to, self.departure, self.arrival,
            getattr(self, 'partner', None)
        )


class IATAFlight(Flight):
    local_departure = None
    local_arrival = None
    localized_departure = None
    localized_arrival = None
    no_departure = None
    no_arrival = None
    code_tariff = None

    _json_attrs = dict(Flight._json_attrs.items() + {
        'local_departure': datetime_converter,
        'local_arrival': datetime_converter,
        'company_iata': asis_converter,
        'station_from_iata': asis_converter,
        'station_to_iata': asis_converter,
        'no_departure': asis_converter,
        'no_arrival': asis_converter,
        'baggage': asis_converter,
        'baggage_norm': asis_converter,
        'avia_company': IfExistsConverter(ModelConverter(AviaCompany)),
        'code_tariff': asis_converter,
        'company_tariff': IfExistsConverter(ModelConverter(CompanyTariff)),
    }.items())

    def make_flight_tag(self):
        f = self
        tag_text = '|'.join('%r' % j for j in [
            f.station_from_iata,
            f.station_to_iata,
            f.local_departure,
            f.local_arrival,
            f.number,
            f.klass,
            f.charter,
        ])
        return hashlib.md5(tag_text).hexdigest()

    def preprocess(self):
        COMPANY_IATA_REPLACEMENTS = {
            'FV': 'SU',
        }

        def process_company_iata(val):
            return COMPANY_IATA_REPLACEMENTS.get(val, val)

        try:
            company_code, num = self.number.split(None, 1)

        except Exception:
            pass

        else:
            company_code = process_company_iata(company_code)
            self.number = '%s %s' % (company_code, num)

        if hasattr(self, 'company_iata'):
            self.company_iata = process_company_iata(self.company_iata)

    def add_to(self, reference):
        reference.add_company(self.company_iata)
        reference.add_station(iata=self.station_from_iata)
        reference.add_station(iata=self.station_to_iata)

    def complete_from(self, reference):
        self.complete_company(reference)

        station_from, station_to = self.complete_stations(reference)

        self.complete_times(station_from, station_to, reference)

    def complete_company(self, reference):
        cache_key = (self.company_iata, self.number)

        if cache_key not in reference.cache:
            reference.cache[cache_key] = reference.company(
                self.company_iata, flight_number=self.number
            )

        self.company = reference.cache[cache_key]

        # Заменим код авиакомпании в номере рейса
        if self.company:
            c = self.company
            number_parts = self.number.split()
            number_parts[0] = c.iata or c.sirena_id or c.icao
            self.number = ' '.join(number_parts)

            try:
                self.avia_company = reference.aviacompanies_by_rasp_company_id[c.id]
            except KeyError:
                log.warning(
                    u'No aviacompany for company: [{}] {}'.format(c.id, c.title)
                )
            else:
                self.complete_by_code_tariff(reference)

    def complete_by_code_tariff(self, reference):
        assert self.company
        assert self.avia_company

        def get_tariff_by_code(code):
            try:
                tariffs = reference.tariffs_by_aviacompany_id[self.avia_company.pk]
            except KeyError:
                log.warning('No CompanyTariff for %s', self.avia_company)
                return None
            default_tariff = None
            for company_tariff in tariffs:
                if not company_tariff.mask:
                    default_tariff = company_tariff
                    continue
                try:
                    if re.match(company_tariff.mask, self.code_tariff or ''):
                        return company_tariff
                except Exception as e:
                    log.error(
                        'CompanyTariff matching error: %r %s. code_tariff: %r',
                        e, company_tariff, self.code_tariff
                    )
            return default_tariff

        self.company_tariff = get_tariff_by_code(self.code_tariff)

    def complete_stations(self, reference):
        self.station_from = reference.station(iata=self.station_from_iata)
        self.station_to = reference.station(iata=self.station_to_iata)

        return self.station_from, self.station_to

    def complete_departure_from_local(self, station_from, reference):
        cache_key = (station_from.id, self.local_departure)
        if cache_key not in reference.cache:
            reference.cache[cache_key] = station_from.localize(loc=self.local_departure)

        self.departure = reference.cache[cache_key]

    def complete_arrival_from_local(self, station_to, reference):
        cache_key = (station_to.id, self.local_arrival)
        if cache_key not in reference.cache:
            reference.cache[cache_key] = station_to.localize(loc=self.local_arrival)

        self.arrival = reference.cache[cache_key]

    def complete_departure_from_localized(self, station_from):
        self.local_departure = station_from.local_time(self.localized_departure).replace(tzinfo=None)
        self.complete_departure_from_local(station_from)

    def complete_arrival_from_localized(self, station_to):
        self.local_arrival = station_to.local_time(self.localized_arrival).replace(tzinfo=None)
        self.complete_arrival_from_local(station_to)

    def complete_times(self, station_from, station_to, reference):
        if station_from:
            if self.local_departure:
                self.complete_departure_from_local(station_from, reference)
            elif self.localized_departure:
                self.complete_departure_from_localized(station_from)

        if station_to:
            if self.local_arrival:
                self.complete_arrival_from_local(station_to, reference)
            elif self.localized_arrival:
                self.complete_arrival_from_localized(station_to)


class UkrmintransTrainRoute(Flight):
    display_t_code = 'train'

    first_station = None
    last_station = None

    _json_attrs = dict(Flight._json_attrs.items() + {
        'first_station': ModelConverter(Station),
        'last_station': ModelConverter(Station),
        'electronic_ticket': asis_converter,
    }.items())

    def make_flight_tag(self):
        f = self
        tag_text = '|'.join('%r' % j for j in [
            f.local_departure,
            f.local_arrival,
            f.number,
            f.ukrmintrans_first_station_code,
            f.ukrmintrans_last_station_code,
            f.electronic_ticket,
        ])
        return hashlib.md5(tag_text).hexdigest()

    def add_to(self, reference):
        reference.add_code_system_station(
            self.ukrmintrans_code_from, 'express'
        )
        reference.add_code_system_station(
            self.ukrmintrans_code_to, 'express'
        )

        reference.add_code_system_station(
            self.ukrmintrans_first_station_code, 'express'
        )
        reference.add_code_system_station(
            self.ukrmintrans_last_station_code, 'express'
        )

    def complete_from(self, reference):
        self.station_from = reference.get_code_system_station(
            self.ukrmintrans_code_from, 'express'
        )
        self.station_to = reference.get_code_system_station(
            self.ukrmintrans_code_to, 'express'
        )

        self.first_station = reference.get_code_system_station(
            self.ukrmintrans_first_station_code, 'express'
        )
        self.last_station = reference.get_code_system_station(
            self.ukrmintrans_last_station_code, 'express'
        )

        self.departure = self.point_tz_from.localize(loc=self.local_departure)
        self.arrival = self.point_tz_to.localize(loc=self.local_arrival)

    def points_for_title(self):
        first_station = self.first_station or self.station_from
        last_station = self.last_station or self.station_to

        return {
            'from': first_station.settlement if first_station.settlement
            else first_station,
            'to': last_station.settlement if last_station.settlement
            else last_station,
        }


class BusRoute(Flight):
    display_t_code = 'bus'
    show_number = False

    _json_attrs = dict(Flight._json_attrs.items() + {
        'show_number': asis_converter,
    }.items())


class UkrmintransBusRoute(BusRoute):
    supplier_code = 'ukrmintrans'

    first_station = None
    last_station = None

    _json_attrs = dict(Flight._json_attrs.items() + {
        'first_station': ModelConverter(Station),
        'last_station': ModelConverter(Station),
    }.items())

    def make_flight_tag(self):
        f = self
        tag_text = '|'.join('%r' % j for j in [
            f.local_departure,
            f.local_arrival,
            f.number,
            f.ukrmintrans_first_station_code,
            f.ukrmintrans_last_station_code,
            f.electronic_ticket,
        ])
        return hashlib.md5(tag_text).hexdigest()

    def add_to(self, reference):
        reference.add_code_system_station(
            self.ukrmintrans_code_from, 'ukrmintrans'
        )
        reference.add_code_system_station(
            self.ukrmintrans_code_to, 'ukrmintrans'
        )

        reference.add_code_system_station(
            self.ukrmintrans_first_station_code, 'ukrmintrans'
        )
        reference.add_code_system_station(
            self.ukrmintrans_last_station_code, 'ukrmintrans'
        )

    def complete_from(self, reference):
        self.station_from = reference.get_code_system_station(
            self.ukrmintrans_code_from, 'ukrmintrans'
        )
        self.station_to = reference.get_code_system_station(
            self.ukrmintrans_code_to, 'ukrmintrans'
        )

        self.first_station = reference.get_code_system_station(
            self.ukrmintrans_first_station_code, 'ukrmintrans'
        )

        self.last_station = reference.get_code_system_station(
            self.ukrmintrans_last_station_code, 'ukrmintrans'
        )

        if self.station_from and self.station_to:
            self.departure = self.station_from and \
                self.station_from.localize(loc=self.local_departure)

            self.arrival = self.station_to and \
                self.station_to.localize(loc=self.local_arrival)

    def points_for_title(self):
        first_station = self.first_station or self.station_from
        last_station = self.last_station or self.station_to

        return {
            'from': first_station.settlement if first_station.settlement
            else first_station,
            'to': last_station.settlement if last_station.settlement
            else last_station,
        }


class BusComUaRoute(BusRoute):
    supplier_code = 'buscomua'

    def make_flight_tag(self):
        f = self
        tag_text = '|'.join('%r' % j for j in [
            f.buscomua_server_code,
            f.buscomua_city_code_from,
            f.buscomua_station_code_from,
            f.buscomua_city_code_to,
            f.buscomua_station_code_to,
            f.local_departure,
            f.local_arrival,
            f.number,
            f.electronic_ticket,
        ])
        return hashlib.md5(tag_text).hexdigest()

    @property
    def station_from_key(self):
        return (self.buscomua_server_code, self.buscomua_city_code_from, self.buscomua_station_code_from)

    @property
    def station_to_key(self):
        return (self.buscomua_server_code, self.buscomua_city_code_to, self.buscomua_station_code_to)

    def add_to(self, reference):
        reference.add_station(buscomua=self.station_from_key)
        reference.add_station(buscomua=self.station_to_key)

    def complete_from(self, reference):
        self.station_from = reference.station(buscomua=self.station_from_key)
        self.station_to = reference.station(buscomua=self.station_to_key)

        if self.station_from and self.station_to:
            self.departure = self.station_from.localize(loc=self.local_departure)
            self.arrival = self.station_to.localize(loc=self.local_arrival)


class SwdfactoryRoute(BusRoute):
    supplier_code = 'swdfactory'

    def make_flight_tag(self):
        f = self
        tag_text = '|'.join('%r' % j for j in [
            f.departure_text,
            f.arrival_text,
            f.local_departure,
            f.local_arrival,
            f.swdfactory_code_from,
            f.swdfactory_code_to,
            f.number,
        ])
        return hashlib.md5(tag_text).hexdigest()

    def add_to(self, reference):
        reference.add_code_system_station(
            self.swdfactory_code_from, 'swdfactory'
        )

        reference.add_code_system_station(
            self.swdfactory_code_to, 'swdfactory'
        )

    def complete_from(self, reference):
        self.station_from = reference.get_code_system_station(
            self.swdfactory_code_from, 'swdfactory'
        )

        self.station_to = reference.get_code_system_station(
            self.swdfactory_code_to, 'swdfactory'
        )

        self.departure = self.station_from and \
            self.station_from.localize(loc=self.local_departure)

        self.arrival = self.station_to and \
            self.station_to.localize(loc=self.local_arrival)

        # заменим для сопоставления рейсов
        self.number = self.departure.strftime('time%H%M') \
            if self.departure is not None else ''


class BlablacarRoute(Flight):
    display_t_code = 'blablacar'

    departure = None
    arrival = None
    local_departure = None
    local_arrival = None

    number_of_variants = None
    duration_in_seconds = None

    _json_attrs = dict(Flight._json_attrs.items() + {
        'number_of_variants': asis_converter,
        'duration_in_seconds': asis_converter,
    }.items())

    def make_flight_tag(self):
        f = self
        tag_text = '|'.join('%r' % j for j in [
            f.departure,
            f.arrival,
            f.station_from,
            f.station_to,
            f.number_of_variants,
            f.duration_in_seconds,
        ])
        return hashlib.md5(tag_text).hexdigest()

    def add_to(self, reference):
        pass

    def complete_from(self, reference):
        pass


class Flights(transfers.Variant, JsonSerializable):
    _json_attrs = {
        'segments': ListConverter(Flight.get_converter()),
    }

    def __init__(self, **kwargs):
        self.display_info = {}
        self.segments = []

    def key(self):
        return "-".join(f.number for f in self.segments)

    def check_transfers_times(self):
        return all(arr.local_arrival <= dep.local_departure
                   for arr, dep in pairwise(self.segments))

    @property
    def local_departure(self):
        return self.segments[0].local_departure if self.segments else None

    @property
    def local_arrival(self):
        return self.segments[-1].local_arrival if self.segments else None


class Variant(JsonSerializable):
    raw_data = ""
    tag = None

    _json_attrs = {
        'forward': Flights.get_converter(),
        'backward': Flights.get_converter(),
        'order_data': asis_converter,
        'klass': asis_converter,
        'tag': asis_converter,
        'tariff': tariff_converter,
        'partner': ModelConverter([Partner, DohopVendor, AmadeusMerchant]),
        'charter': asis_converter,

        'raw_seats': asis_converter,
        'raw_tariffs': DictConverter(asis_converter, tariff_converter),
        'raw_is_several_prices': asis_converter,
    }

    def __init__(self, **kwargs):
        self.forward = Flights()
        self.backward = Flights()
        self.klass = None
        self.order_data = {}
        self.display_info = {}

        self.raw_seats = None
        self.raw_tariffs = dict()
        self.raw_is_several_prices = dict()

    def make_tag(self):
        tag_data = [
            '|'.join(s.make_flight_tag() for s in self.forward.segments),
            '|'.join(s.make_flight_tag() for s in self.backward.segments),
            self.klass,
            self.tariff.value,
            self.partner.code,
        ]

        tag_text = '@'.join('%r' % j for j in tag_data)
        return hashlib.md5(tag_text).hexdigest()

    def booking_available(self, now):
        if not self.forward.segments[0].departure:
            return True

        departure = self.forward.segments[0].departure

        if not departure.tzinfo:
            departure = MSK_TZ.localize(departure)

        partner = getattr(self, 'partner', None)
        hours = settings.PARTNERS_BOOKING_AVAILABLE_HOURS.get(
            partner.code if partner else None,
            2
        )

        if departure and now > departure - timedelta(hours=hours):
            return False

        return True

    @property
    def completed_ok(self):
        try:
            if self.backward.segments and \
                    self.backward.departure and self.forward.arrival and \
                    self.backward.departure - self.forward.arrival < \
                    settings.MIN_STAY_TIME:
                raise BadVariantException(
                    'Backward departure is earlier than forward arrival')

        except BadVariantException as e:
            self.what_completed_bad = str(e)

            return False

        return True

    def bad(self, q):
        try:
            self.check_bad(q)

        except BadVariantException as e:
            self.what_bad = str(e)

            return True

        return False

    def check_bad(self, q):
        if not self.forward:
            raise BadVariantException('No forward')

        if q.date_backward and not (self.backward and self.backward.segments):
            raise BadVariantException('No backward for roundtrip')

        if not self.forward.check_transfers_times():
            raise BadVariantException('Bad forward transfers times')

        if self.backward.segments and not self.backward.check_transfers_times():
            raise BadVariantException('Bad backward transfers times')

        if self.backward.segments and \
                self.backward.local_departure and self.forward.local_arrival and \
                self.backward.local_departure - self.forward.local_arrival < \
                settings.MIN_STAY_TIME:
            raise BadVariantException(
                'Backward departure is earlier than forward arrival')

        priorities = self.get_service_priorities()

        if priorities and min(priorities) < SERVICE_CLASS_PRIORITIES[q.klass]:
            raise BadVariantException('Segments service class does not match')

        if self.klass and self.klass in SERVICE_CLASS_PRIORITIES and \
                q.klass in SERVICE_CLASS_PRIORITIES and \
                SERVICE_CLASS_PRIORITIES[self.klass] < \
                SERVICE_CLASS_PRIORITIES[q.klass]:
            raise BadVariantException('Variant service class does not match')

        # Если дата вылета туда или обратно не равна запрашиваемой
        if self.forward.segments[0].local_departure and \
                self.forward.segments[0].local_departure.date() != q.date_forward:
            raise BadVariantException('Date forward does not match')

        if self.backward and self.backward.segments[0].local_departure and \
                self.backward.segments[0].local_departure.date() != q.date_backward:
            raise BadVariantException('Date backward does not match')

        if self.tariff.value <= 0:
            raise BadVariantException('Zero or negative price')

    @property
    def is_charter(self):
        return getattr(self, 'charter', False)

    def get_service_priorities(self):
        return filter(None, map(SERVICE_CLASS_PRIORITIES.get, self.klasses))

    def black_listed(self, rates, query_black_list):
        def black_rule_match(self, rule, rates):
            any_condition_hit = False

            # PARTNER
            if rule.partner:
                if rule.partner == getattr(self, 'partner', None):
                    any_condition_hit = True
                else:
                    return False

            backward = getattr(self, 'backward', None)

            forward_settlement_from = getattr(self.forward.segments[0].station_from, "settlement", None)
            forward_settlement_to = getattr(self.forward.segments[-1].station_to, "settlement", None)

            backward_settlement_from = getattr(self.backward.segments[0].station_from, "settlement", None) if backward else None
            backward_settlement_to = getattr(self.backward.segments[-1].station_to, "settlement", None) if backward else None

            forward_country_from = getattr(forward_settlement_from, "country", None)
            forward_country_to = getattr(forward_settlement_to, "country", None)

            backward_country_from = getattr(backward_settlement_from, "country", None) if backward else None
            backward_country_to = getattr(backward_settlement_to, "country", None) if backward else None

            # FROM
            if rule.station_from:
                if rule.station_from_id == self.forward.segments[0].station_from.id or \
                        (getattr(self, 'backward', None) and rule.station_from_id == self.backward.segments[0].station_from.id):
                    any_condition_hit = True
                else:
                    return False

            if rule.settlement_from:
                if rule.settlement_from == forward_settlement_from or (backward and rule.settlement_from == backward_settlement_from):
                    any_condition_hit = True
                else:
                    return False

            if rule.country_from:
                if rule.country_from == forward_country_from or (backward and rule.country_from == backward_country_from):
                    any_condition_hit = True
                else:
                    return False

            # TO
            if rule.station_to:
                if rule.station_to_id == self.forward.segments[-1].station_to.id or \
                        (getattr(self, 'backward', None) and rule.station_to_id == self.backward.segments[-1].station_to.id):
                    any_condition_hit = True
                else:
                    return False

            if rule.settlement_to:
                if rule.settlement_to == forward_settlement_to or (backward and rule.settlement_to == backward_settlement_to):
                    any_condition_hit = True
                else:
                    return False

            if rule.country_to:
                if rule.country_to == forward_country_to or (backward and rule.country_to == backward_country_to):
                    any_condition_hit = True
                else:
                    return False

            if rule.currency:
                if rule.currency.code != self.tariff.currency:
                    tariff = copy.copy(self.tariff)

                    if rates and tariff.currency in rates and rule.currency.code in rates:
                        tariff.base_value = float(tariff.value) * float(rates[tariff.currency]) / float(rates[rule.currency.code])
                        tariff.currency = str(rule.currency.code)
                        tariff.value = tariff.base_value
                    else:
                        tariff = None

                else:
                    tariff = self.tariff

                if tariff:
                    if rule.price_from and rule.price_to:
                        if rule.price_from <= tariff.value <= rule.price_to:
                            any_condition_hit = True
                        else:
                            return False

                    elif rule.price_from:
                        if rule.price_from <= tariff.value:
                            any_condition_hit = True
                        else:
                            return False

                    elif rule.price_to:
                        if tariff.value <= rule.price_to:
                            any_condition_hit = True
                        else:
                            return False

            if rule.company and rule.flight_number:
                company_iata_and_flight_number_hit = False

                for segment in self.all_segments:
                    company = getattr(segment, 'company', None)

                    if company and rule.company_id == company.id and rule.flight_number == segment.number.split(' ')[-1]:
                        company_iata_and_flight_number_hit = True
                        break

                if company_iata_and_flight_number_hit:
                    any_condition_hit = True

                else:
                    return False

            elif rule.company:
                company_iata_hit = False

                for segment in self.all_segments:
                    company = getattr(segment, 'company', None)

                    if company and int(rule.company.id) == int(company.id):
                        company_iata_hit = True
                        break

                if company_iata_hit:
                    any_condition_hit = True

                else:
                    return False

            elif rule.flight_number:
                flight_number_hit = False
                for segment in self.all_segments:
                    if rule.flight_number == segment.number.split(' ')[-1]:
                        flight_number_hit = True
                        break

                if flight_number_hit:
                    any_condition_hit = True

                else:
                    return False

            if rule.when_from and rule.when_to:
                if rule.when_from <= self.forward.segments[0].local_departure.date() and rule.when_to >= self.forward.segments[-1].local_arrival.date():
                    any_condition_hit = True

                else:
                    return False

            elif rule.when_from:
                if rule.when_from <= self.forward.segments[0].local_departure.date():
                    any_condition_hit = True

                else:
                    return False

            elif rule.when_to:
                if rule.when_to >= self.forward.segments[-1].local_arrival.date():
                    any_condition_hit = True
                else:
                    return False

            if rule.klass:
                if rule.klass == self.klass:
                    any_condition_hit = True

                else:
                    return False

            if any_condition_hit:
                return True

            return False

        if self.forward.segments[0].t_type.code != 'plane':
            return False

        for rule in query_black_list:
            if black_rule_match(self, rule, rates):
                return True

            else:
                continue

    @property
    def all_segments(self):
        return self.forward.segments + (self.backward.segments or [])

    @property
    def klasses(self):
        return set([s.klass for s in self.all_segments if s.klass])

    @property
    def by_flights_key(self):
        return "-".join(f.number for f in self.all_segments)

    @property
    def klass_name(self):
        return CLS_NAMES.get(self.klass, u'эконом')

    def __str__(self):
        try:
            return '%s[%s](%s) %s [%s]' % (
                self.__class__.__name__,
                getattr(self, 'partner', None),
                self.tariff,
                self.klass,
                self.by_flights_key,
            )
        except Exception:
            log.exception('Flight.__str__')
            return repr(self)

    def __repr__(self):
        return "<%s [%s]>" % (
            self.__class__.__name__,
            getattr(self, 'partner', None)
        )
