# -*- coding: utf-8 -*-
import logging
from datetime import datetime
from time import time as clock_time
from urllib import quote_plus

from django.conf import settings

from travel.avia.library.python.avia_data.models import AmadeusMerchant
from travel.avia.library.python.common.models.geo import Country, Station, Settlement
from travel.avia.library.python.common.models.partner import Partner, DohopVendor
from travel.avia.library.python.common.models_utils.geo import Point
from travel.avia.library.python.common.utils.caching import cache_method_result
from travel.avia.library.python.common.utils.date import parse_date

from travel.avia.avia_api.avia.lib.serialization import (
    JsonPackable, ModelConverter, IfExistsConverter, asis_converter, date_converter,
)

log = logging.getLogger(__name__)


class QueryIsNotValid(Exception):
    pass


class ApiDTError(Exception):
    pass


class QueryAllException(Exception):
    pass


point_converter = ModelConverter(
    [Station, Settlement],
    {
        'title': asis_converter,  # MAYBE unused
        'point_key': asis_converter,
    }
)


class Query(JsonPackable):
    _json_attrs = {
        'id': asis_converter,
        'created': asis_converter,
        'national_version': asis_converter,
        'service': asis_converter,
        'user_settlement': IfExistsConverter(point_converter),
        'point_from': point_converter,
        'point_to': point_converter,
        'date_forward': date_converter,
        'date_backward': date_converter,
        'klass': asis_converter,
        'passengers': asis_converter,
        'store': asis_converter,
        'search_url': IfExistsConverter(asis_converter),
        't_code': IfExistsConverter(asis_converter),
    }

    country_code_from = None
    country_code_to = None
    user_settlement = None
    search_url = None
    t_code = None
    partners = ()  # Immutable list. Attr for backward compatability
    service = None

    def __init__(
        self,
        point_from,
        point_to,
        passengers,
        date_forward,
        date_backward=None,
        klass='economy',
        national_version='ru',
        t_code='plane',
        service=None,
        **kwargs
    ):
        self.point_from = point_from
        self.point_to = point_to
        self.passengers = passengers
        self.klass = klass
        self.date_forward = date_forward
        self.date_backward = date_backward
        self.national_version = national_version
        self.t_code = t_code
        self.service = service or getattr(settings, 'PROJECT_CODE', None)

        if not self.service:
            raise Exception('Either "service" should be provided to Query() '
                            'or PROJECT_CODE filled')

        if not self.point_from:
            raise ApiDTError('Empty point_from')

        if not self.point_to:
            raise ApiDTError('Empty point_to')

        for k, v in kwargs.iteritems():
            setattr(self, k, v)

        self.store = False

        self.created = clock_time()

        self.id = '{:%y%m%d-%H%M%S}-{:0>3d}:{}:{}'.format(
            datetime.fromtimestamp(int(self.created)),
            # Миллисекунды времени создания
            int(1000 * (self.created % 1)),
            self.service,
            self.key()
        )

    def fill(self):
        if self.point_from:
            self.fill_codes('from')

        if self.point_to:
            self.fill_codes('to')

    @property
    def qid_msg(self):
        return '(qid:%s)' % self.id

    def key(self):
        params = (
            self.point_from.point_key,
            self.point_to.point_key,
            str(self.date_forward),
            str(self.date_backward),
            self.klass,
            str(self.passengers.get('adults', 0)),
            str(self.passengers.get('children', 0)),
            str(self.passengers.get('infants', 0)),
            self.national_version,
        )
        key = '_'.join([str(p) for p in params])

        return key

    @classmethod
    def from_key(cls, key, service=None):
        (
            point_from_key,
            point_to_key,
            date_forward,
            date_backward,
            klass,
            adults,
            children,
            infants,
            national_version
        ) = key.split('_')[:9]

        adults = int(adults)
        children = int(children)
        infants = int(infants)

        return cls(
            point_from=Point.get_any_by_key(point_from_key),
            point_to=Point.get_any_by_key(point_to_key),
            date_forward=parse_date(date_forward),
            date_backward=None if date_backward == 'None' else parse_date(date_backward),
            klass=klass,
            passengers={
                'adults': int(adults),
                'children': int(children),
                'infants': int(infants),
            },
            national_version=national_version,
            service=service
        )

    @property
    def passengers_count(self):
        passengers = getattr(self, 'passengers', None) or {}

        return sum([
            int(passengers.get('adults', 0)),
            int(passengers.get('children', 0)),
            int(passengers.get('infants', 0)),
        ])

    def _get_rasp_train_partner_codes(self):
        """RASPFRONT-2149"""
        point_from = self.point_from
        point_to = self.point_to
        national_version = self.national_version

        from_country_id = point_from.country_id
        to_country_id = point_to.country_id

        if from_country_id is None or to_country_id is None:
            return []

        # RASP-14147
        if national_version == 'ru':
            if from_country_id == to_country_id == Country.UKRAINE_ID:
                return ['ukrmintrans_train']
            # RASP-8130: не обращаться к УФС если обе станции вне РФ
            if from_country_id == Country.RUSSIA_ID or to_country_id == Country.RUSSIA_ID:
                return ['ufs']

        elif national_version == 'ua':
            if from_country_id == to_country_id == Country.RUSSIA_ID:
                return ['ufs']
            if from_country_id == Country.UKRAINE_ID or to_country_id == Country.UKRAINE_ID:
                return ['ukrmintrans_train']

        return []

    def as_header(self):
        return quote_plus(self.key().encode('ASCII', 'replace'))

    def is_valid(self):
        return hasattr(self, 'iata_from') and self.iata_from \
            and hasattr(self, 'iata_to') and self.iata_to

    CAN_ASK_PARTNER_CACHE_TIME = 3600

    def station_postfilter(self, v):
        """
        Берем только сегементы с совпадающими станциями,
        или со станциями относящимися к городу, напрямую и через station2settlement
        """

        # Оптимизация
        # Большая часть поисков от крупного города до крупного
        # Мск, спб, екб и тд, в которых много станций.
        # Чтобы за ними не ходить быстро проверим а не совпадают ли города в варианте
        # с городами которые искал пользователь
        stations_from = [v.forward.segments[0].station_from]
        stations_to = [v.forward.segments[-1].station_to]

        if v.backward.segments:
            stations_from.append(v.backward.segments[-1].station_to)
            stations_to.append(v.backward.segments[0].station_from)

        # Варианты без точек отправления и прибытия считаем невалидными сразу
        if not all(stations_from) or not all(stations_to):
            return False

        # Выходим только если совпали, иначе нужно проверять как раньше
        if (all([s.settlement == self.point_from for s in stations_from]) and
                all([s.settlement == self.point_to for s in stations_to])):
            return True

        allowed_stations_from = self.get_allowed_stations(self.point_from)
        allowed_stations_to = self.get_allowed_stations(self.point_to)

        if allowed_stations_from:
            st = v.forward.segments[0].station_from
            if not st or st.id not in allowed_stations_from:
                return False

            if v.backward.segments:
                st = v.backward.segments[-1].station_to
                if not st or st.id not in allowed_stations_from:
                    return False

        if allowed_stations_to:
            st = v.forward.segments[-1].station_to
            if not st or st.id not in allowed_stations_to:
                return False

            if v.backward.segments:
                st = v.backward.segments[0].station_from
                if not st or st.id not in allowed_stations_to:
                    return False

        return True

    @cache_method_result
    def get_allowed_stations(self, point):
        if isinstance(point, Station):
            return [point.id]

        if not isinstance(point, Settlement):
            return None

        settlement = point

        stations = list(settlement.station_set.values_list('id', flat=True))
        stations.extend(settlement.station2settlement_set.values_list('station', flat=True))

        return stations

    def fill_codes(self, direction):
        point = getattr(self, 'point_{}'.format(direction))

        self.fill_iata_codes(point, direction)

    def fill_iata_codes(self, point, direction):
        iata = point.iata or point.iata_point and point.iata_point.iata

        if iata:
            setattr(self, 'iata_{}'.format(direction), iata)
            setattr(self, 'iata_real_{}'.format(direction), iata)

        else:
            sirena = point.sirena_id or \
                point.sirena_point and point.sirena_point.sirena_id

            setattr(self, 'iata_{}'.format(direction), sirena or None)

        if isinstance(point, Country):
            setattr(self, 'country_code_{}'.format(direction), point.code)

        elif hasattr(point, 'country') and isinstance(point.country, Country):
            setattr(self, 'country_code_{}'.format(direction),
                    point.country.code)

    def __repr__(self):
        return '<Query: %s>' % self.id


def get_actual_partners(national_version, t_code='plane', from_rasp=False, mobile=False):
    partners = Partner.get_actual(
        t_code, national_version, from_rasp=from_rasp, mobile=mobile
    )

    # Если Дохоп в списке, то добавим и его вендоров
    if any(p.code == 'dohop' for p in partners):
        partners = [p for p in partners if p.code != 'dohop']
        partners += DohopVendor.get_actual(national_version, mobile=mobile)

    # Если Амадеус в списке, то добавим и его AmadeusMerchant
    if any(p.code == 'amadeus' for p in partners):
        partners = [p for p in partners if p.code != 'amadeus']
        partners += AmadeusMerchant.get_actual(national_version, mobile=mobile)

    return partners


def not_mauch_than(count, text):
    if len(text) <= count:
        return text
    suffix = '...'
    if count < len(suffix):
        return count[:count]
    return text[:count - len(suffix)] + suffix
