# -*- coding: utf-8 -*-
from __future__ import unicode_literals

import hashlib
import logging

from django.conf import settings
from flask import Flask
from flask_cors import CORS
from raven.contrib.flask import Sentry
from werkzeug.exceptions import HTTPException, default_exceptions

from travel.avia.avia_api.avia.lib.helpers import get_time_id
from travel.avia.avia_api.avia.lib.jsend import ApiFail, JsendError, JsendFail
from travel.avia.avia_api.avia.lib.api_json import ApiJSONEncoder
from travel.avia.avia_api.avia.lib.sessions import AviaSessionInterface
from travel.avia.avia_api.avia.v1.views import import_all_views
from travel.avia.avia_api.avia.application.base import configure_app


log = logging.getLogger(__name__)


def create_app(config=None):
    app = Flask(__name__)
    CORS(
        app,
        resources=r'/*',
        origins=r'https?://(www\.)?(.*\.)?yandex\.(ru|by|ua|kz|com|com\.tr|az)',
    )

    app = configure_app(app, config)

    app.json_encoder = ApiJSONEncoder
    app.config['JSONIFY_PRETTYPRINT_REGULAR'] = False

    app.session_interface = AviaSessionInterface()

    import_all_views()
    from travel.avia.avia_api.avia.v1.views.base import bp_api_v1, bp_api_v2
    app.register_blueprint(bp_api_v1, url_prefix='/v1.0')
    app.register_blueprint(bp_api_v2, url_prefix='/v2.0')

    from travel.avia.avia_api.avia.v1.internal.api import internal_blueprint
    app.register_blueprint(internal_blueprint, url_prefix='/internal/v1')

    from travel.avia.avia_api.avia.v1.sovetnik.api import sovetnik_blueprint
    app.register_blueprint(sovetnik_blueprint, url_prefix='/sovetnik/v1.0')

    from travel.avia.avia_api.avia.version import bp as bp_version
    app.register_blueprint(bp_version, url_prefix='/version')

    from travel.avia.avia_api.avia.v1.email_dispenser.api import email_subscription_blueprint
    app.register_blueprint(
        email_subscription_blueprint, url_prefix='/email_subscription/v1.0'
    )

    from travel.avia.avia_api.avia.v1.takeout.api import takeout_blueprint
    app.register_blueprint(
        takeout_blueprint, url_prefix='/takeout/v1.0'
    )

    from travel.avia.avia_api.avia.v1.afl_search.api import afl_search_blueprint
    app.register_blueprint(afl_search_blueprint, url_prefix='/aeroflot-search/v1.0')

    if settings.SENTRY_DSN:
        app.config['SENTRY_DSN'] = settings.SENTRY_DSN
        Sentry(app)

    @app.errorhandler(ApiFail)
    def handle_api_fail(api_fail):
        return api_fail.get_response()

    def handle_error(e):
        if isinstance(e, HTTPException):
            response = JsendFail(
                {'description': e.description}, e.code
            ).get_response()

        else:
            error_id = '{}-{}'.format(
                get_time_id(),
                hashlib.md5(repr(e)).hexdigest()[:5]
            )

            log.exception('%s %s', error_id, repr(e))

            response = JsendError(
                'Error id: {}'.format(error_id)
            ).get_response()

            if isinstance(e, HTTPException):
                response.status_code = e.code

        return response

    for code in default_exceptions.iterkeys():
        app.errorhandler(code)(handle_error)

    return app
