from __future__ import unicode_literals

import logging
from datetime import datetime

from bson import ObjectId

from travel.avia.avia_api.avia.lib.passport_utils import get_yandex_uid_by_oauth_token
from travel.avia.avia_api.avia.lib.pusher import Pusher
from travel.avia.avia_api.avia.v1.model.device import Device
from travel.avia.avia_api.avia.v1.model.user import User

log = logging.getLogger(__name__)


def get_yandex_uid(oauth_token, remote_address):
    if oauth_token:
        yandex_uid = get_yandex_uid_by_oauth_token(oauth_token, remote_address)
        log.info('%s -> %s', oauth_token, yandex_uid)

        if not yandex_uid:
            log.warning('Bad oauth_token %r', oauth_token)

        return yandex_uid

    return None


def get_user_by_hello_data(uuid, yandex_uid):
    if yandex_uid:
        User.objects(
            yandex_uid=yandex_uid
        ).update(
            unset__uuid=True,
            set__touched=datetime.utcnow(),
            upsert=True
        )

        user = User.objects.get(yandex_uid=yandex_uid)

        user.absorb_users(User.objects(uuid=uuid))
        return user

    User.objects(uuid=uuid).update(
        unset__yandex_uid=None,
        set__touched=datetime.utcnow(),
        upsert=True
    )

    return User.objects.get(uuid=uuid)


def get_device_by_hello_data(device_id, uuid, lang, country, platform,
                             push_token, yandex_uid):
    device = None
    pusher = Pusher()
    transport = pusher.transport().name

    if device_id:
        try:
            device = Device.objects.get(id=ObjectId(device_id))
        except Exception as exc:
            log.critical("Device with id %r isn't found: %r", device_id, exc)
        else:
            if device.uuid != uuid:
                if device.is_first_winphone_launch():
                    log.info("Wphone uuid change: %r -> %r", device.uuid, uuid)
                    try:
                        device.user.update(uuid=uuid)  # TODO: remove
                    except Exception as exc:
                        log.warning(
                            "Couldn't update user's %r uuid to %r: %r",
                            device.user, uuid, exc
                        )
                    else:
                        device.user.reload()
                else:
                    log.critical(
                        "Device %r changes its uuid! %r -> %r",
                        device, device.uuid, uuid
                    )

                try:
                    device.update(uuid=uuid)
                except Exception as exc:
                    log.warning(
                        "Device with id=%r couldn't change its uuid to %r: %r",
                        device.id, uuid, exc
                    )
                else:
                    device.reload()

    if uuid:
        try:
            device = Device.objects.get(uuid=uuid)
        except Exception as exc:
            log.info("Device with uuid %r isn't found: %r", uuid, exc)

    user = get_user_by_hello_data(uuid, yandex_uid)

    if device is None:
        Device.objects(uuid=uuid).update(
            user=user,
            lang=lang,
            country=country,
            platform=platform,
            transport=transport,
            push_token=push_token,
            upsert=True
        )
        device = Device.objects.get(uuid=uuid)
    else:
        device.user.update(pull__devices=device)
        device.update(
            user=user,
            lang=lang,
            country=country,
            platform=platform,
            transport=transport,
            push_token=push_token,
        )
        device.reload()

    user.update(add_to_set__devices=device)

    if push_token:
        pusher.add(device)

    return device


def hello(
    session,
    uuid, oauth_token, push_token, platform, device_time, lang, country,
    currency, national_version, remote_address, known_device_id=None,
):
    device = None
    yandex_uid = get_yandex_uid(oauth_token, remote_address)

    try:
        device = get_device_by_hello_data(
            device_id=known_device_id,
            uuid=uuid,
            lang=lang,
            country=country,
            platform=platform,
            push_token=push_token,
            yandex_uid=yandex_uid,
        )
    except Exception as exc:
        log.critical('Device creation error: %r', exc, exc_info=True)

    if not device:
        log.critical("Couldn't create a device for uuid %r", uuid)

    new_session = session.copy()
    new_session.update({
        'device_id': device.id if device else None,
        'uuid': uuid,
        'push_token': push_token,
        'device_time': device_time,
        'lang': lang,
        'country': country,
        'national_version': national_version,
        'remote_addr': remote_address,
        'session_ttl': 60 * 60 * 24,
        'yandex_uid': yandex_uid,
        'currency': currency
    })

    return new_session
