# coding=utf-8
from collections import defaultdict

from travel.avia.library.python.common.models.geo import (
    Settlement, Station, StationType, Station2Settlement,
)

from travel.avia.avia_api.avia.cache.utils import by_title_cache, normalize


class SettlementCache(object):
    def __init__(self):
        self._by_id = {}
        self._by_title = {}
        self._by_station_id = {}

    def precache(self):
        # Здесь нельзя использовать .order_by до тех пор, пока у Settlement
        # не будет оторван PrecachingManager. После прекеша order_by ломается.
        settlements = tuple(Settlement.objects.all())
        self._by_id = {s.id: s for s in settlements}
        self._by_title = by_title_cache(settlements)

        by_station_id = defaultdict(list)

        for station_id, settlement_id in Station.objects.filter(
            settlement_id__isnull=False
        ).values_list(
            'id', 'settlement_id'
        ):
            by_station_id[station_id].append(settlement_id)

        for relation in Station2Settlement.objects.all():
            by_station_id[relation.station_id].append(relation.settlement_id)

        self._by_station_id = {
            station_id: tuple(
                self._by_id[id]
                for id in settlement_ids
                if id in self._by_id
            )
            for station_id, settlement_ids in by_station_id.iteritems()
        }

    def by_title(self, title):
        return self._by_title.get(normalize(title)) or ()

    def by_station_id(self, station_id):
        return self._by_station_id.get(station_id, tuple())

    def by_id(self, pk):
        # type: (int) -> Settlement
        return self._by_id.get(pk, None)


class AviaSettlements(object):
    def __init__(self, settlement_cache):
        self._origin = settlement_cache
        self._avia_ids = frozenset()

    def precache(self):
        airports = tuple(Station.objects.filter(
            station_type_id=StationType.AIRPORT_ID,
        ))

        self._avia_ids = frozenset(
            Settlement.objects.filter(
                id__in=[
                    airport.settlement_id for airport in airports
                ] + [
                    relation.settlement_id
                    for relation in Station2Settlement.objects.filter(
                        station__in=airports
                    )
                ],
            ).values_list('id', flat=True)
        )

    def by_title(self, title):
        return tuple(
            settlement
            for settlement in self._origin.by_title(title)
            if settlement.id in self._avia_ids
        )

    def by_station_id(self, station_id):
        return tuple(
            settlement
            for settlement in self._origin.by_station_id(station_id)
            if settlement.id in self._avia_ids
        )


settlement_cache = SettlementCache()
avia_settlement_cache = AviaSettlements(settlement_cache)
