# coding=utf-8
from collections import defaultdict

from travel.avia.library.python.common.models.geo import Station, StationType

from travel.avia.avia_api.avia.cache.utils import by_title_cache, normalize


class StationCache(object):
    def __init__(self):
        self._by_id = {}
        self._by_title = {}

    def precache(self):
        # Здесь нельзя использовать .order_by до тех пор, пока у Station
        # не будет оторван PrecachingManager. После прекеша order_by ломается.
        stations = tuple(Station.objects.all())
        self._by_id = {station.id: station for station in stations}
        self._by_title = by_title_cache(stations)

    def by_id(self, station_id):
        # type: (int) -> Station
        return self._by_id.get(station_id)

    def by_title(self, title):
        return self._by_title.get(normalize(title)) or ()


class Airports(object):
    def __init__(self, station_cache):
        self._origin = station_cache
        self._airport_ids = frozenset()
        self._by_settlement_geo_id = defaultdict(list)

    def precache(self):
        airports = Station.objects.filter(station_type_id=StationType.AIRPORT_ID)
        self._airport_ids = frozenset(
            airports.values_list('id', flat=True)
        )
        self._by_settlement_geo_id = defaultdict(list)
        for airport in tuple(airports):
            if airport.settlement_id and not airport.hidden:
                self._by_settlement_geo_id[airport.settlement._geo_id].append(airport.id)

    def by_id(self, airport_id):
        result = self._origin.by_id(airport_id)
        if result and result.id in self._airport_ids:
            return result
        return None

    def by_title(self, title):
        return tuple(
            station
            for station in self._origin.by_title(title)
            if station.id in self._airport_ids
        )

    def by_settlement_geo_id(self, settlement_geo_id):
        # type: (int) -> list[int]
        return self._by_settlement_geo_id.get(settlement_geo_id)


station_cache = StationCache()
airport_cache = Airports(station_cache)
