# -*- coding: utf-8 -*-
import logging
from operator import attrgetter

from collections import defaultdict

from travel.avia.library.python.common.models.translations import TranslatedTitle
from travel.avia.library.python.common.models_utils.i18n import new_Geobase_L_title

LANGUAGES = ('ru', 'en', 'uk', 'tr')
GEO_BASE_LANGUAGE = 'be'

logger = logging.getLogger(__name__)


def normalize(title):
    title = ''.join(title.split())
    return title.lower().strip().replace(u'ё', u'е').replace('-', '')


def by_title_cache(points):
    translations_by_id = {
        translation['id']: translation
        for translation in TranslatedTitle.objects
        .filter(id__in=map(attrgetter('new_L_title_id'), points))
        .values('id', 'ru_nominative', 'en_nominative', 'uk_nominative',
                'tr_nominative')
    }

    by_title = defaultdict(set)

    for s in points:
        for lang in LANGUAGES:
            translation_model = translations_by_id.get(s.new_L_title_id)
            if not translation_model:
                continue
            translation = (
                translation_model.get('%s_nominative' % lang, '')
                or _get_geobase_translation(s, lang)
            )
            if not translation:
                continue

            by_title[normalize(translation)].add(s)

    sort_key = attrgetter('majority_id', 'id')

    return {
        title: tuple(sorted(models, key=sort_key))
        for title, models in by_title.iteritems()
    }


def _get_geobase_translation(point, lang):
    if not isinstance(point.L_title.field, new_Geobase_L_title):
        return None
    try:
        return point.L_title.field.exact_geobase_linguistic(
            point, lang, 'nominative'
        )
    except Exception as e:
        logger.warning(
            'Could not read translation from geobase for point %s, lang %s in NOM case.\n%s',
            point,
            lang,
            repr(e),
        )
