# -*- coding: utf-8 -*-
from __future__ import unicode_literals

from datetime import datetime, timedelta
import logging
import time

from django.conf import settings
from travel.avia.contrib.python.mongoengine.mongoengine.errors import NotUniqueError

from travel.avia.avia_api.avia.lib.mongo_iterator import iterate_chunks
from travel.avia.avia_api.avia.lib.push_notifications import (
    AeroexpressPushNotification, MinPricePushNotification
)
from travel.avia.avia_api.avia.lib.cache import JsonFileCache
from travel.avia.avia_api.avia.v1.model.flight import Flight
from travel.avia.avia_api.avia.v1.model.favorite import MinPriceNotification
from travel.avia.avia_api.avia.v1.model.aeroexpress import Aeroexpress
from travel.avia.avia_api.avia.v1.model.user import User
from travel.avia.avia_api.avia.v1.model.distributed_lock import DistributedLock
from travel.avia.avia_api.avia.v1.views.info import collect_raw_por, serialize_por

log = logging.getLogger(__name__)


def cache_por_for_languages(langs):
    log.info('Collect raw POR')
    try:
        raw_por = collect_raw_por()
    except Exception as exc:
        log.exception("Couldn't create POR: %r", exc)
        raise

    error = False
    for lang in langs:
        log.info('Updating POR[%s]', lang)

        try:
            data = serialize_por(raw_por, lang)
            por_cache = JsonFileCache(settings.POR_CACHE_FILENAME[lang])
            if not por_cache.exists() or data != por_cache.load():
                por_cache.save(data)
            else:
                log.info("The cache for [%s] didn't change", lang)
        except Exception as exc:
            log.exception(
                "Couldn't serialize POR[%s]: %r", lang, exc
            )
            error = True

    if error:
        raise Exception("Some PORs couldn't be serialized")


def push_aeroexpress_notifications():
    log.info('Pushing aeroexpress notifications')
    start_time = time.time()

    try:
        _push_aeroexpress_notifications()
    except Exception as exc:
        log.exception(
            'Unexpected error in push_aeroexpress_notifications : %r', exc
        )
    else:
        elapsed = time.time() - start_time
        log.info('Pushed aeroexpress notifications: %s seconds', int(elapsed))


def _push_aeroexpress_notifications():
    now = datetime.utcnow()

    # начинаем рассылать уведомления за полтора часа
    datetime_before = now + timedelta(minutes=90)

    # заканчиваем за час до отправления аэроэкспресса
    datetime_after = now + timedelta(minutes=60)

    flights_with_errors = []

    while True:
        flight = Flight.objects(
            aeroexpress__notification_status=Aeroexpress.NOTIFICATION_WAITING,
            aeroexpress__datetime__gt=datetime_after,
            aeroexpress__datetime__lte=datetime_before,
        ).modify(
            aeroexpress__notification_status=Aeroexpress.NOTIFICATION_SENDING
        )

        if flight is None:
            break

        if flight.canceled:
            flight.update(aeroexpress__notification_status=Aeroexpress.NOTIFICATION_CANCELLED)
            continue

        try:
            users = list(User.objects(aeroex_notify_flights=flight))
            AeroexpressPushNotification(flight).send(users)
        except Exception as exc:
            log.exception(
                "Error while pushing aeroexpress notification: %r", exc
            )
            flights_with_errors.append(flight.id)
            continue
        else:
            log.info('Pushed aeroexpress notification for %r', flight)

        flight.update(
            aeroexpress__notification_status=Aeroexpress.NOTIFICATION_SENT
        )

    if flights_with_errors:
        Flight.objects(id__in=flights_with_errors).update(
            aeroexpress__notification_status=Aeroexpress.NOTIFICATION_WAITING
        )


MINPRICE_THRESHOLD_PERCENT = 0.5
NOTIFICATION_MIN_PAUSE = 60 * 60 * 4


def iterate_users():
    # Нельзя итерироваться по всей коллекции сразу, так как протухает курсор: RASPTICKETS-4736
    # Нельзя выкачивать всех юзеров в память, так как их очень много.
    # Берём только юзеров, активных за последний год.
    for user in iterate_chunks(
        User, 'id',
        User.objects.filter(touched__gte=datetime.utcnow() - timedelta(days=365))
    ):
        yield user


def update_min_prices():
    lock = DistributedLock(
        name='update_min_prices',
        expire_after=datetime.utcnow() + timedelta(minutes=30)
    )

    try:
        lock.save()
    except NotUniqueError:
        log.info('Other process is already updating minimal prices')
        return

    log.info('Started the update of minimal prices')

    try:
        for user in iterate_users():
            if not user.favorites:
                continue

            try:
                user.reload()
            except Exception as exc:
                log.exception(
                    "Couldn't reload user [%r]: %s. User fields: %r",
                    user, exc, user.to_json()
                )
                continue

            for favorite in user.favorites:
                try:
                    if not favorite.is_relevant():
                        log.info(
                            'User\'s [%s] favorite [%s] isn\'t relevant',
                            user, favorite,
                        )
                        continue

                    favorite.update_min_price(user)

                    notify_user_if_min_price_has_changed(user, favorite)

                except Exception as exc:
                    log.exception(
                        "Error while handling user's [%r] min price for his "
                        "favorite [%s]: %s. User fields: %r",
                        user, favorite, exc, user.to_json()
                    )
            try:
                user.save()
            except Exception as exc:
                log.exception(
                    'Saving user error %r: %s. User fields: %r',
                    user, exc, user.to_json(),
                )
            else:
                log.info('Saved user %r', user)

    except Exception:
        log.exception('Got an error while updating minimal prices')
    finally:
        lock.delete()

    log.info('Successfully updated minimal prices')


def notify_user_if_min_price_has_changed(user, favorite):
    prev_price = favorite.last_min_price_notification.price

    log.info(
        "Checking user's [%r] min price change for favorite [%s]: %s -> %s",
        user, favorite, prev_price.value, favorite.min_price.value
    )

    try:
        change = (
            favorite.min_price.value - prev_price.value
        ) / prev_price.value

    except ZeroDivisionError:
        return

    if abs(change) * 100 < MINPRICE_THRESHOLD_PERCENT:
        log.info('Change %r is too low', change)
        return

    prev_when = favorite.last_min_price_notification.when

    if prev_when is not None:
        delta = datetime.utcnow() - prev_when
        elapsed = delta.total_seconds()

        if elapsed < NOTIFICATION_MIN_PAUSE:
            log.info(
                "It's only been %d seconds since the last notification",
                elapsed
            )
            return

    log.info(
        'Notifying user [%s] about the new price for his favorite [%s]: %s',
        user, favorite, favorite.min_price.value
    )

    MinPricePushNotification(favorite).send([user])

    favorite.last_min_price_notification = MinPriceNotification(
        price=favorite.min_price,
        when=datetime.utcnow()
    )
