# -*- coding: utf-8 -*-
import json
import logging
from datetime import timedelta

import requests
import ujson
from django.conf import settings
from requests.exceptions import ReadTimeout

from travel.avia.library.python.common.utils import marketstat

from travel.avia.avia_api.avia.http_api_schemas import SearchResults
from travel.avia.avia_api.avia.lib.decorators import skip_None_values

log = logging.getLogger(__name__)

yt_search_log = marketstat.DsvLog(settings.YT_SEARCH_LOG_PATH)


class DaemonBadRequest(Exception):
    pass


class TicketDaemon(object):
    def __init__(self, url=None):
        self._url = url or settings.DAEMON_URL

    def _jsend(self, method, params):
        try:
            r = requests.get(
                '{}/jsendapi/{}/'.format(self._url, method),
                params={
                    param: (
                        json.dumps(value)
                        if isinstance(value, (list, dict, tuple)) else
                        value
                    )
                    for param, value in params.iteritems()
                },
                timeout=settings.DAEMON_API_TIMEOUT,
                verify=False
            )
            log.info('Request to %r took %fs.', method, r.elapsed.total_seconds())
        except ReadTimeout:
            log.error('Read timeout on ticket-daemon-API %r request', method)
            raise
        if not r.ok:
            exc_class = DaemonBadRequest if r.status_code == 400 else Exception
            raise exc_class(
                'TD request error({}): %r'.format(r.status_code), r.content
            )
        return ujson.loads(r.content)['data']

    def init_search(
        self,
        point_from, point_to, date_forward, date_backward,
        adults, children, infants,
        flight_class, national_version, lang,
        country_geo_id, service='api_avia', ignore_cache=False,
    ):
        return self._jsend('avia_api/init_search', {
            'point_from': point_from.point_key,
            'point_to': point_to.point_key,
            'date_forward': date_forward.strftime('%Y-%m-%d'),
            'date_backward': (
                date_backward.strftime('%Y-%m-%d') if date_backward else None
            ),
            'klass': flight_class,
            'adults': adults,
            'children': children,
            'infants': infants,
            'national': national_version,
            'lang': lang,
            'service': service,
            'ignore_cache': ignore_cache,
            'client_ip_country_geo_id': country_geo_id,
        })['qid']

    # workaround before using the http-daemon fully
    def front_qid(
        self, point_from, point_to,
        date_forward, date_backward,
        adults, children, infants,
        klass, national_version, lang, service,
    ):
        return str(
            self._jsend('avia_api/redirect_page_qid', {
                'point_from': point_from.point_key,
                'point_to': point_to.point_key,
                'date_forward': date_forward.strftime('%Y-%m-%d'),
                'date_backward': (
                    date_backward.strftime('%Y-%m-%d')
                    if date_backward else None
                ),
                'adults': adults,
                'children': children,
                'infants': infants,
                'klass': klass,
                'national': national_version,
                'lang': lang,
                'service': service,
            })
        )

    def search_results(
        self, qid, lang, currency,
        skip_partners=None, tags=None, result_pages=None, clid=None,
    ):
        return SearchResults.from_json(
            self._jsend('avia_api/results', skip_None_values({
                'qid': qid,
                'lang': lang,
                'currency': currency,
                'skip': skip_partners or [],
                'result_pages': result_pages or {},
                'tags': tags,
                'clid': clid,
            }))
        )

    def redirect_data(
        self, qid, order_data, lang, yandex_uid=None, passport_uid=None, user_ip=None
    ):
        data = self._jsend('avia_api/redirect_data', {
            'qid': qid,
            'lang': lang,
            'order_data': order_data,
            'user_info': skip_None_values({
                'yandexuid': yandex_uid,
                'passportuid': passport_uid,
                'userip': user_ip,
            }),
        })
        return {
            'url': data['url'],
            'post': data.get('post'),
        }

    def yaclid_age(self, raw_yaclid):
        return timedelta(
            seconds=self._jsend(
                'decode_yaclid', {'yaclid': raw_yaclid}
            )['age_seconds']
        )


class LoggingTD(object):
    def __init__(self, daemon):
        self._daemon = daemon

    def init_search(
        self,
        point_from, point_to, date_forward, date_backward,
        adults, children, infants,
        flight_class, national_version, lang,
        country_geo_id, service='api_avia', ignore_cache=False,
        user_ip=None, user_geoid=None, user_passportuid=None,
        user_platform=None, user_uuid=None,
    ):
        yt_search_log.write_event(
            passportuid=user_passportuid,
            userip=user_ip,
            extra={
                'tskv_format': 'rasp-users-search-log',
                'service': {
                    'api_avia': 'api-app',
                    'sovetnik': 'api-sovetnik',
                    'tours': 'api-tours',
                }[service],
                'metrica_api_uuid': user_uuid,
                'pp': user_platform,
                'geoid': user_geoid,

                'from_id': point_from.point_key,
                'to_id': point_to.point_key,
                'adults': adults,
                'children': children,
                'infants': infants,
                'when': date_forward.strftime('%Y-%m-%d'),
                'return_date': (
                    date_backward.strftime('%Y-%m-%d')
                    if date_backward else None
                ),
                'klass': flight_class,
                'national_version': national_version,
            }
        )

        return self._daemon.init_search(
            point_from, point_to, date_forward, date_backward,
            adults, children, infants,
            flight_class, national_version, lang,
            country_geo_id, service, ignore_cache,
        )
