# -*- coding: utf-8 -*-
from __future__ import unicode_literals

import logging
from urlparse import urljoin

import requests
from django.conf import settings

from travel.library.python.tvm_ticket_provider import provider_fabric

APIKEYS_TVM_ALIAS = 'balance-apikeys'


log = logging.getLogger(__name__)


class ApiKeysError(Exception):
    pass


if settings.ENABLE_TVM:
    apikeys_tvm_provider = provider_fabric.create(
        fake=settings.FAKE_API_KEYS,
        source_id=settings.TVM_NAME,
        destinations=[APIKEYS_TVM_ALIAS],
        renew=True,
    )


def check_key(key, user_ip=None):
    try:
        if settings.FAKE_API_KEYS:
            return _fake_check_key(key, user_ip)

        return _check_key(key, user_ip)

    except Exception as exc:
        log.error('Check key error %r', exc)

        raise ApiKeysError(exc)


def _check_key(key, user_ip=None):
    url = urljoin(settings.APIKEYS_URL, 'check_key')

    params = {
        'key': key,
        'user_ip': user_ip or '',
        'service_token': settings.APIKEYS_SERVICE_TOKEN,
    }

    try:
        r = requests.get(
            url,
            headers={
                'X-Ya-Service-Ticket': apikeys_tvm_provider.get_ticket(APIKEYS_TVM_ALIAS),
            },
            params=params,
            timeout=settings.APIKEYS_TIMEOUT
        )
    except Exception as e:
        raise Exception('Query error: %r %r %r', url, params, e)

    if not r.ok:
        raise Exception('Response http error: %r %r %r' %
                        (url, r.status_code, r.content))

    try:
        d = r.json()
    except Exception:
        raise Exception('Response json error. %r', r.content)

    if d['result'] != 'OK':
        raise Exception('Response data error. Result not OK: %r', d)

    return d['key_info']


def _fake_check_key(key, user_ip=None):
    key = '_'.join(key.split())
    platforms = sorted(settings.BILLING_PLATFORMS.keys())
    year = 2016
    years = range(year - len(platforms), year + 1)
    keynum = sum(ord(c) for c in key)
    platform = platforms[keynum % len(platforms)]
    year = years[keynum % len(years)]
    name = '%s %s' % (platform, year)

    return {
        'hidden': False,
        'dt': '%s-02-17:13:24.000000' % year,
        'user': {
            'name': 'Fake_name',
            'roles': [],
            'balance_client_id': None,
            'login': 'fake_login',
            'email': 'fake_login@yandex.ru',
            'uid': 204800000 + keynum,
        },
        'name': name,
        'id': key,
    }


def create_key(uid):
    if not settings.DEBUG:
        raise Exception('Do not create keys in production')

    r = requests.get(
        urljoin(settings.APIKEYS_URL, 'create_key'),
        headers={
            'X-Ya-Service-Ticket': apikeys_tvm_provider.get_ticket(APIKEYS_TVM_ALIAS),
        },
        params={
            'user_uid': uid,
            'service_token': settings.APIKEYS_SERVICE_TOKEN,
        },
        timeout=settings.APIKEYS_TIMEOUT
    )

    if not r.ok:
        raise Exception('%r %r' % (r.status_code, r.content))

    return r.json()['result']['key']
