# -*- coding: utf-8 -*-
from __future__ import unicode_literals

import logging
from datetime import datetime, timedelta

from django.conf import settings
from flask import session
from flask_login import UserMixin

from travel.avia.library.python.common.utils.date import MSK_TZ

from travel.avia.avia_api.avia.lib.apikeys import ApiKeysError
from travel.avia.avia_api.avia.lib.date import get_utc_now, get_msk_now
from travel.avia.avia_api.avia.lib.descriptor import DictProxyDescriptor
from travel.avia.avia_api.avia.lib import apikeys

log = logging.getLogger(__name__)


DT_FMT = '%Y-%m-%d_%H:%M:%S'


class AppUser(UserMixin):
    def __init__(self, app_key):
        self.app_key = app_key
        self.created = get_msk_now()

    uuid = DictProxyDescriptor(session, 'uuid')
    yandex_uid = DictProxyDescriptor(session, 'yandex_uid')
    oauth_token = DictProxyDescriptor(session, 'oauth_token')
    push_token = DictProxyDescriptor(session, 'push_token')
    platform = DictProxyDescriptor(session, 'platform')
    session_ttl = DictProxyDescriptor(session, 'session_ttl')
    remote_addr = DictProxyDescriptor(session, 'remote_addr')  # IPv4 or IPv6
    device_id = DictProxyDescriptor(session, 'device_id')
    lang = DictProxyDescriptor(session, 'lang')
    country = DictProxyDescriptor(session, 'country')
    national_version = DictProxyDescriptor(session, 'national_version')
    currency = DictProxyDescriptor(session, 'currency')

    @property
    def age(self):
        return get_msk_now() - self.created

    @property
    def id(self):
        return '%s %s' % (self.created.astimezone(MSK_TZ).strftime(DT_FMT),
                          self.app_key)

    @classmethod
    def load(cls, id_):
        try:
            created_raw, app_key = id_.split(None, 1)
            created = MSK_TZ.localize(datetime.strptime(created_raw, DT_FMT))

            appuser = cls(app_key)
            appuser.created = created

        except Exception as exc:
            log.exception('AppUser.load error: %r', exc)
            return None

        if appuser.age.total_seconds() > appuser.session_ttl:
            log.info(
                'Session expired: %r. Age: %r. Created: %r. Ttl: %r',
                appuser, appuser.age, appuser.created, appuser.session_ttl
            )

            if not settings.DEBUG:
                return None

            log.debug('Continue with expired session')

        return appuser

    @classmethod
    def get_by_app_key(cls, app_key):
        try:
            key_name = apikeys.check_key(app_key)['name']
        except ApiKeysError as exc:
            log.critical('Apikeys error: %r', exc)
            platform = settings.FALLBACK_PLATFORM
        else:
            platform = platform_by_keyname(key_name)

        appuser = cls(app_key)
        appuser.platform = platform
        return appuser

    @property
    def time_diff(self):
        return timedelta(seconds=session.get('offset', 0))

    @time_diff.setter
    def time_diff(self, delta):
        session['offset'] = delta.total_seconds()

    @property
    def device_time(self):
        return get_utc_now() + self.time_diff

    @device_time.setter
    def device_time(self, dt):
        self.time_diff = dt - get_utc_now()

    def __repr__(self):
        return (
            '<{} Uuid: {}, YandexUid: {}, Created: {}, '
            'AppKey: {}, Platform: {}, Ip: {}, DeviceId: {}>'.format(
                self.__class__.__name__,
                self.uuid,
                self.yandex_uid,
                self.created,
                self.app_key,
                self.platform,
                self.remote_addr,
                self.device_id,
            )
        )


def platform_by_keyname(keyname):
    log.info('key %r', keyname)
    try:
        platform = keyname.strip().split()[0].lower()

        if platform not in settings.BILLING_PLATFORMS:
            raise Exception('Unknown platform: %r' % platform)

    except Exception as exc:
        log.warning('Keyname parse error: %r %r', keyname, exc)

        return None

    return platform
