# -*- coding: utf-8 -*-
from __future__ import unicode_literals

from logging import getLogger

from django.conf import settings

from travel.avia.library.python.avia_data.libs.currency.rates.caching import CurrencyRatesCache
from travel.avia.library.python.common.models.currency import Currency as RaspCurrency, Price as CommonPrice

log = getLogger(__name__)


class CurrencyRatesInfo(object):
    def __init__(self, source_name, rates):
        self._source = source_name
        self._rates = rates

    def source(self):
        return self._source

    def rates(self):
        return self._rates.copy()


class CurrencyRatesInfoStorage(object):
    def info(self, national_version):
        rates_cache = CurrencyRatesCache.load(national_version)
        if rates_cache is not None:
            source_name = rates_cache.src
            rates = rates_cache.rates
        else:
            log.warning('Rasp currency rates are being used')
            source_name, rates = RaspCurrency.fetch_rates(
                RaspCurrency.objects.all(),
                settings.NATIONAL_VERSIONS_CAPITAL_GEOID[national_version],
                settings.AVIA_NATIONAL_CURRENCIES[national_version]
            )

        if settings.FAKE_CURRENCY_RATES:
            if source_name is None and len(rates) == 1:
                return rates.keys()[0], rates

        return CurrencyRatesInfo(source_name, rates)


class Price(CommonPrice):
    def __init__(self, value, currency=None):
        currency = {'RUB': 'RUR'}.get(currency, currency)

        return super(Price, self).__init__(value, currency)

    def __add__(self, other):
        if not isinstance(other, CommonPrice):
            raise ValueError(
                u'Can only sum prices with prices. Got %s' % type(other))

        if self.currency != other.currency:
            raise ValueError(u'Can only add prices with same currency')

        return Price(self.value + other.value, self.currency)

    def __eq__(self, other):
        if not isinstance(other, CommonPrice):
            return False
        if self.currency != other.currency:
            return False
        return self.value == other.value

    @classmethod
    def convert_to_national(cls, tariff, national_version, rates):
        try:
            to_currency = settings.AVIA_NATIONAL_CURRENCIES[national_version]
        except KeyError:
            raise ValueError('Not alowed national_version [%s]',
                             national_version)

        if tariff.currency == to_currency:
            return tariff

        if not rates:
            raise ValueError('No rates')

        if tariff.currency not in rates:
            raise ValueError('Tariff currency [%s] not in rates',
                             tariff.currency)

        if to_currency not in rates:
            raise ValueError('Target currency [%s] not in rates',
                             to_currency)

        return cls(
            float(tariff.value) *
            float(rates[tariff.currency]) /
            float(rates[to_currency]),
            to_currency
        )
