# -*- coding: utf-8 -*-
import email.utils
import logging
import os
from datetime import date, datetime, timedelta

import pytz

from travel.avia.library.python.common.utils.date import MSK_TZ

from travel.avia.avia_api.avia.lib.decorators import refuse_None

log = logging.getLogger(__name__)


def awarify(obj):
    if obj.tzinfo:
        return obj

    return MSK_TZ.localize(obj)


def get_utc_now():
    return pytz.UTC.localize(datetime.utcnow())


def get_msk_now():
    return get_utc_now().astimezone(MSK_TZ)


DT_FMT_LOCAL = '%Y-%m-%d %H:%M:%S'
LEN_DT_FMT_LOCAL = len(datetime.now().strftime(DT_FMT_LOCAL))


def parse_dt_local(text):
    return datetime.strptime(text[:LEN_DT_FMT_LOCAL], DT_FMT_LOCAL)


def aware_dt_from_timestamp(timestamp):
    naive_utc_dt = datetime.utcfromtimestamp(timestamp)
    return pytz.utc.localize(naive_utc_dt)


@refuse_None
def format_aware_dt_utc(dt):
    return dt.astimezone(pytz.UTC).strftime('%Y-%m-%d %H:%M:%S UTC')


DT_BROWSER_FMT = '%a, %d %b %Y %H:%M:%S'


def format_aware_dt_for_browser(dt):
    return dt.astimezone(pytz.UTC).strftime(DT_BROWSER_FMT) + ' GMT'


def parse_rfc2822_datetime(text):
    date_tuple = email.utils.parsedate_tz(text)
    if not date_tuple:
        log.critical(
            'Parsing HTTP header error: %r', text
        )
        return None

    return datetime.fromtimestamp(
        email.utils.mktime_tz(date_tuple),
        tz=pytz.UTC
    )


def test_formatting_rfc822_dt():
    for now in [get_utc_now(), get_msk_now()]:
        expected = now.replace(microsecond=0)
        actual = parse_rfc2822_datetime(format_aware_dt_for_browser(now))
        assert actual == expected

# File times


def modification_datetime(filename):
    return aware_dt_from_timestamp(os.path.getmtime(filename))


def inodealter_datetime(filename):
    return aware_dt_from_timestamp(os.path.getctime(filename))


def test_inodealter_datetime():
    import random
    import time

    filename = '/tmp/unused_file_inode_%s' % random.random()

    if os.path.exists(filename):
        os.unlnk(filename)

    delta = timedelta(seconds=.5)
    begin = get_utc_now()

    with open(filename, 'w') as f:
        pass

    inode_dt = inodealter_datetime(filename)

    assert begin - delta/2 <= inode_dt
    assert inode_dt < begin + delta/2

    time.sleep(delta.total_seconds())
    # Writing existed file changes its inode altering time
    with open(filename, 'w') as f:
        pass
    assert inode_dt < inodealter_datetime(filename)

    inode_dt = inodealter_datetime(filename)
    time.sleep(delta.total_seconds())
    # Reading file not changes its inode altering time
    with open(filename, 'r') as f:
        f.read()
    assert inode_dt == inodealter_datetime(filename)

    # Chmod file changes its inode altering time
    os.chmod(filename, os.stat(filename).st_mode)
    assert inode_dt < inodealter_datetime(filename)

    os.unlink(filename)


def test_modification_datetime():
    import random
    import time

    filename = '/tmp/unused_file_%s' % random.random()
    delta = timedelta(seconds=.5)

    def iteration():
        begin = get_utc_now()

        with open(filename, 'w'):
            pass

        mod_dt = modification_datetime(filename)
        assert begin - delta/2 <= mod_dt
        assert mod_dt < begin + delta/2

    # Writing existing file changes its modification time
    iteration()
    time.sleep(delta.total_seconds())
    iteration()

    # Reading file does not changes its modification time
    mod_dt = modification_datetime(filename)
    with open(filename, 'r') as f:
        f.read()
    assert mod_dt == modification_datetime(filename)

    os.unlink(filename)


@refuse_None
def datepart(date_like):
    return date(date_like.year, date_like.month, date_like.day)
