# -*- coding: utf-8 -*-
from functools import update_wrapper, wraps
from inspect import getargspec

from decorator import decorator


# Copy of django.utils.decorators.method_decorator
def method_decorator(decorator):
    """
    Converts a function decorator into a method decorator
    """

    # 'func' is a function at the time it is passed to _dec, but will
    # eventually be a method of the class it is defined it.
    def _dec(func):
        def _wrapper(self, *args, **kwargs):
            @decorator
            def bound_func(*args2, **kwargs2):
                return func(self, *args2, **kwargs2)

            # bound_func has the signature that 'decorator' expects i.e.  no
            # 'self' argument, but it is a closure over self so it can call
            # 'func' correctly.
            return bound_func(*args, **kwargs)

        # In case 'decorator' adds attributes to the function it decorates, we
        # want to copy those. We don't have access to bound_func in this scope,
        # but we can cheat by using it on a dummy function.

        @decorator
        def dummy(*args, **kwargs):
            pass

        update_wrapper(_wrapper, dummy)
        # Need to preserve any existing attributes of 'func',
        # including the name.
        update_wrapper(_wrapper, func)

        return _wrapper

    update_wrapper(_dec, decorator)
    # Change the name to aid debugging.
    _dec.__name__ = 'method_decorator(%s)' % decorator.__name__
    return _dec


def maybe_method_decorator(dec):
    m_dec = method_decorator(dec)

    @wraps(dec)
    def dec_wrapper(fn):
        spec = getargspec(fn)

        if spec.args and spec.args[0] == 'self':
            return m_dec(fn)

        else:
            return dec(fn)

    return dec_wrapper


@maybe_method_decorator
def elementwise(fn):
    @wraps(fn)
    def wrapper(arg, *args, **kwargs):
        # is a Sequence
        if hasattr(arg, '__getitem__') and not isinstance(arg, basestring):
            return [fn(item, *args, **kwargs) for item in arg]

        else:
            return fn(arg, *args, **kwargs)

    return wrapper


def _skip_None_values(d):
    if isinstance(d, dict):
        # Type(d) allow work with dict subclasses and return same type
        return type(d)(
            filter(lambda (key, val): val is not None, d.items())
        )

    return d


@maybe_method_decorator
@decorator
def dec_skip_None_values(fn, *args, **kwargs):
    return _skip_None_values(fn(*args, **kwargs))


def skip_None_values(obj, *args, **kwargs):
    if isinstance(obj, dict):
        return _skip_None_values(obj, *args, **kwargs)

    # Act as function decorator
    elif callable(obj):
        fn = obj

        if args or kwargs:
            raise ValueError

        return dec_skip_None_values(fn)

    raise ValueError


@maybe_method_decorator
@decorator
def refuse_None(fn, *args, **kwargs):
    if args and args[0] is None:
        return None

    return fn(*args, **kwargs)


def pipe_out(process_results_fn):
    @maybe_method_decorator
    @decorator
    def deco_pipe(fn, *args, **kwargs):
        return process_results_fn(fn(*args, **kwargs))

    return deco_pipe
