# -*- coding: utf-8 -*-
import logging
from datetime import datetime

from enum import Enum

from travel.avia.avia_api.avia.v1.model.flight_status import FlightStatusActual, FlightStatusCancel

log = logging.getLogger(__name__)


class FlightInfo(object):
    def update_flight(self, flight):
        raise NotImplementedError


class WizardFlightStatus(Enum):
    Wait = 'wait'
    Late = 'late'
    Departure = 'departure'
    Arrive = 'arrive'
    Cancelled = 'cancelled'
    Unknown = 'unknown'
    Nodata = 'nodata'


class WizardFlightInfo(FlightInfo):
    def __init__(
        self,
        number,  # str
        scheduled_departure_datetime,  # aware datetime
        scheduled_arrival_datetime,  # aware datetime
        real_departure_datetime,  # aware datetime
        company_id,  # int
        status,  # WizardFlightStatus or None
        status_title,  # str
        from_station_id,  # int
        from_settlement_geoid,  # int
        from_iata,  # station iata
        to_station_id,
        to_settlement_geoid,
        to_iata,
    ):
        assert scheduled_departure_datetime.tzinfo
        assert scheduled_arrival_datetime.tzinfo
        assert real_departure_datetime.tzinfo

        self.number = number
        self.scheduled_departure_datetime = scheduled_departure_datetime
        self.scheduled_arrival_datetime = scheduled_arrival_datetime
        self.real_departure_datetime = real_departure_datetime
        self.company_id = company_id
        self.status = status
        self.status_title = status_title
        self.from_station_id = from_station_id
        self.from_settlement_geoid = from_settlement_geoid
        self.from_iata = from_iata
        self.to_station_id = to_station_id
        self.to_settlement_geoid = to_settlement_geoid
        self.to_iata = to_iata

    def departure_station_key(self):
        return 's{}'.format(self.from_station_id)

    def arrival_station_key(self):
        return 's{}'.format(self.to_station_id)

    def update_flight(self, flight):
        flight.departure_date = datetime(
            year=self.scheduled_departure_datetime.year,
            month=self.scheduled_departure_datetime.month,
            day=self.scheduled_departure_datetime.day
        )

        flight.departure_datetime = (
            self.scheduled_departure_datetime.replace(tzinfo=None)
        )

        flight.arrival_datetime=(
            self.scheduled_arrival_datetime.replace(tzinfo=None)
        )

        flight.company_id = self.company_id
        flight.departure = self.departure_station_key()
        flight.arrival = self.arrival_station_key()

        if self.status == WizardFlightStatus.Cancelled:
            if not flight.canceled:
                try:
                    flight.update_status(FlightStatusCancel())
                except Exception as exc:
                    log.exception(
                        'Error while trying to cancel flight %r: %r',
                        flight, exc
                    )

        departs_at = self.real_departure_datetime.replace(tzinfo=None)
        try:
            if departs_at > flight.any_departure_dt:
                flight.update_status(
                    FlightStatusActual(actual_datetime=departs_at)
                )
            elif departs_at < flight.any_departure_dt:
                log.warning(
                    "Updating flight's %r actual datetime to an earlier time",
                    flight
                )
        except Exception:
            log.exception(
                "Actualizing flight's %r departure time error: %s -> %s",
                flight, flight.any_departure_dt, departs_at
            )


class SearchFlightInfo(FlightInfo):
    def __init__(self,
                 number,
                 scheduled_departure_datetime,  # aware datetime
                 scheduled_arrival_datetime,  # aware datetime
                 company_id,  # int or None
                 departure_point,  # Point or None
                 arrival_point,  # Point or None
                 ):

        assert scheduled_departure_datetime.tzinfo
        assert scheduled_arrival_datetime.tzinfo

        self.number = number
        self.scheduled_departure_datetime = scheduled_departure_datetime
        self.scheduled_arrival_datetime = scheduled_arrival_datetime
        self.company_id = company_id
        self.departure_point = departure_point
        self.arrival_point = arrival_point

    @classmethod
    def from_segment(cls, segment):
        return cls(
            number=segment.number,
            scheduled_departure_datetime=segment.departs_at,
            scheduled_arrival_datetime=segment.arrives_at,
            company_id=segment.company.id if segment.company else None,
            departure_point=segment.departure_station,
            arrival_point=segment.arrival_station,
        )

    def scheduled_departure_date(self):
        return datetime(
            year=self.scheduled_departure_datetime.year,
            month=self.scheduled_departure_datetime.month,
            day=self.scheduled_departure_datetime.day,
        )

    def update_flight(self, flight):
        if self.departure_point is not None:
            flight.departure = self.departure_point.point_key

        if self.arrival_point is not None:
            flight.arrival = self.arrival_point.point_key

        if self.company_id:
            flight.company_id = self.company_id

        flight.departure_datetime = (
            self.scheduled_departure_datetime.replace(tzinfo=None)
        )
        flight.arrival_datetime = (
            self.scheduled_arrival_datetime.replace(tzinfo=None)
        )

        # Можно обойтись и без этого. Только для оптимизации
        flight.cache_airport_from(self.departure_point)
        flight.cache_airport_to(self.arrival_point)
