# -*- coding: utf-8 -*-
import logging
import re
from datetime import datetime
from functools import wraps
from time import time as clock_time

from django.utils.encoding import force_unicode
from flask import current_app, make_response
from itsdangerous import URLSafeSerializer
from werkzeug import BaseResponse
from pytils.translit import detranslify, translify

from travel.avia.avia_api.avia.lib.decorators import maybe_method_decorator
from travel.avia.avia_api.avia.lib.jsend import JsendSuccess, Jsend

log = logging.getLogger(__name__)

RE_LETTER = re.compile(ur'[A-ZА-ЯЁ]', flags=re.UNICODE)


def get_time_id():
    created = clock_time()

    return '{:%y%m%d-%H%M%S}-{:0>3d}'.format(
        datetime.fromtimestamp(int(created)),
        # Миллисекунды времени создания
        int(1000 * (created % 1)),
    )


def get_serializer(secret_key=None, **kwargs):
    if secret_key is None:
        secret_key = current_app.secret_key

    return URLSafeSerializer(secret_key, **kwargs)


def view(app, url, *args, **kwargs):
    super_route = app.route

    defaults = kwargs.pop('defaults', {})
    route_id = object()
    # FIXME how about to do not use _route_id
    defaults['_route_id'] = route_id

    def deco(fn):
        @super_route(url, defaults=defaults, *args, **kwargs)
        @wraps(fn)
        def wrapper(*args2, **kwargs2):
            this_route = kwargs2.get('_route_id')
            if not getattr(fn, 'is_route', False):
                del kwargs2['_route_id']

            result = fn(*args2, **kwargs2)

            if this_route is not route_id:
                return result

            # catch redirects.
            if isinstance(result, (current_app.response_class, BaseResponse)):
                return result

            if isinstance(result, Jsend):
                return result.get_response()

            return JsendSuccess(data=result).get_response()

        wrapper.is_route = True
        return wrapper

    return deco


def add_response_headers(headers):
    """ This decorator adds the headers passed in to the response """
    @maybe_method_decorator
    def deco(fn):
        @wraps(fn)
        def wrapper(*args, **kwargs):
            resp = make_response(fn(*args, **kwargs))
            h = resp.headers
            for header, value in headers.items():
                h[header] = value
            return resp
        return wrapper
    return deco


class Undefined(object):
    def __getattr__(self, name):
        return self

    def __getitem__(self, name):
        return self

    def __iter__(self):
        return iter([])

    def __nonzero__(self):
        return False

    def __json__(self):
        return None


undefined = Undefined()


def wrap(v):
    if isinstance(v, dict):
        return DictWrapper(v)

    if isinstance(v, list):
        return ListWrapper(v)

    return v


class DictWrapper(dict):
    def __getitem__(self, name):
        try:
            return wrap(dict.__getitem__(self, unicode(name)))

        except KeyError:
            return undefined

    def __getattr__(self, name):
        if name.startswith('__'):
            raise AttributeError

        return self[name]

    def __setattr__(self, name, value):
        self[name] = value


class ListWrapper(list):
    def __getitem__(self, i):
        try:
            return wrap(list.__getitem__(self, i))

        except IndexError:
            return undefined

    def __iter__(self):
        for v in list.__iter__(self):
            yield wrap(v)


def fix_flight_number(flight_number):
    # Переведем все в верхний регистр
    fixed_flight_number = flight_number.upper()

    # Удалим пробелы и дефисы
    fixed_flight_number = re.sub(ur'[\s-]+', '', fixed_flight_number, flags=re.UNICODE)

    # Разделим и склеим номер
    fixed_flight_number = '%s %s' % (
        fixed_flight_number[:2],
        fixed_flight_number[2:],
    )

    # Найдем первую букву
    match = re.search(RE_LETTER, fixed_flight_number)

    if not match:
        raise ValueError

    first_airline_code_letter = match.group()

    # Переводим все символы в латиницу, если первый символ латинский
    if ord('A') <= ord(first_airline_code_letter) <= ord('Z'):
        fixed_flight_number = translify(force_unicode(fixed_flight_number))

    # иначе, в кирилицу
    else:
        fixed_flight_number = detranslify(force_unicode(fixed_flight_number))

    # Если номер начинается с кода АК, то уберем код из номера
    flight_number_parths = fixed_flight_number.split(' ')

    if len(flight_number_parths) == 2:
        airline, number = flight_number_parths

        if number.startswith(airline):
            fixed_flight_number = '%s %s' % (
                airline,
                number[len(airline):]
            )

    return fixed_flight_number
