# -*- coding: utf-8 -*-
import logging
import ujson

from django.conf import settings

from travel.avia.library.python.avia_data.models import MinPrice
from travel.avia.library.python.common.models.geo import Settlement, Station

from travel.avia.avia_api.ant.custom_types import HttpSearchIdentificator
from travel.avia.avia_api.avia.api.caches import shared_cache
from travel.avia.avia_api.avia.daemon_http_api import TicketDaemon, LoggingTD
from travel.avia.avia_api.avia.lib.flight_classes import FlightClass
from travel.avia.avia_api.avia.v1.email_dispenser.helpers.mysql_cache import avia_data_currency_cache
from travel.avia.avia_api.avia.v1.email_dispenser.helpers.variants_combiner import combine_variants_data

log = logging.getLogger(__name__)


def get_min_price_by_key(key):
    return shared_cache.get(key)


def get_point_settlement(point):
    if isinstance(point, Settlement):
        return point
    if isinstance(point, Station) and point.settlement:
        if point.settlement is None:
            log.info('Station %r has no settlement', point)
        return point.settlement
    return None


class FlightType(object):
    Direct = 'direct'
    Indirect = 'indirect'


class StoredMinPrice(object):
    def __init__(self, value, currency, stored_variants=None):
        self.value = float(value)
        self.currency = currency
        self.stored_variants = stored_variants


class StoredVariant(object):
    def __init__(self, departure_flight_numbers, return_flight_numbers):
        self.flight_numbers_to = departure_flight_numbers
        self.flight_numbers_from = return_flight_numbers

    def __hash__(self):
        return hash(self._projection)

    def __eq__(self, other):
        return self._projection == other._projection

    @property
    def _projection(self):
        return (
            tuple(self.flight_numbers_from or []),
            tuple(self.flight_numbers_to or [])
        )


class StoredVariants(object):
    def __init__(self, variants):
        self.variants = variants

    @classmethod
    def from_routes(cls, routes):
        """
        Routes example: SU 1,SU2;SU 3/AF 9;AF 9
        Should be parsed into:
        [
            StoredVariant(['SU 1', 'SU 2'], ['AF 9'],
            StoredVariant(['SU 3'], ['AF 9']
        ]
        One-way flights don't have a return part.
        In case there isn't enough information to convert given flight numbers
        to flights, the whole variant direction is assigned to None.
        """
        log.info('Parsing routes [%s]', routes)

        if '/' not in routes:
            routes = u'{}/'.format(routes)

        splitted_variants = [
            [
                one_direction_flights.split(',')
                for one_direction_flights in variant_list.split(';')
            ]
            for variant_list in routes.split('/')
        ]

        flights_to, flights_from = splitted_variants

        if not flights_from:
            flights_from = [None] * len(flights_to)

        return cls(variants=[
            StoredVariant(numbers_to, numbers_from)
            for (numbers_to, numbers_from) in zip(
                flights_to, flights_from
            )
        ])


class BaseMinPriceStorage(object):
    def warm_up(self, passengers):
        raise NotImplementedError

    def get_min_price(self, flight_type, passengers):
        raise NotImplementedError


class MinPriceStorage(BaseMinPriceStorage):
    def __init__(self, point_from, point_to, departure_date, return_date,
                 national_version, currency_code=None):

        self.point_from = point_from
        self.point_to = point_to
        self.departure_date = departure_date
        self.return_date = return_date
        self.national_version = national_version
        self.currency_code = currency_code

    def warm_up(self, passengers):
        try:
            qid = LoggingTD(TicketDaemon(settings.DAEMON_URL)).init_search(
                point_from=self.point_from,
                point_to=self.point_to,
                date_forward=self.departure_date,
                date_backward=self.return_date,
                adults=passengers.adults,
                children=passengers.children,
                infants=passengers.infants,
                flight_class=FlightClass.Economy,
                national_version=self.national_version,
                lang='ru',
                country_geo_id=None,
                service='tours',
            )

            search_id = HttpSearchIdentificator(
                qid=qid,
                national_version=self.national_version,
                point_from_key=self.point_from.point_key,
                point_to_key=self.point_to.point_key,
                date_forward=self.departure_date,
            ).encrypt()
        except Exception as exc:
            log.exception("Couldn't warm up for query: %r", exc)
            return None

        log.info('Initiated a search request: %r', search_id)
        return search_id


class MinPriceYtStorage(object):
    def __init__(self, yt_minprice_variants_provider, yt_popular_variants_provider, qkey):
        """

        :param avia.v1.email_dispenser.helpers.yt_cache.QkeyMinPrice yt_minprice_variants_provider:
        :param avia.v1.email_dispenser.helpers.yt_cache.PopularVariants yt_popular_variants_provider:
        :param basestring qkey:
        """
        self.yt_minprice_provider = yt_minprice_variants_provider
        self.yt_top_variants_provider = yt_popular_variants_provider
        self.qkey = qkey

    def get_min_price(self, _filter=None):
        combined_data = combine_variants_data(
            min_prices=self.yt_minprice_provider.by_qkey.get(self.qkey),
            top_variants=self.yt_top_variants_provider.by_qkey.get(self.qkey),
            _filter=_filter,
        )

        if combined_data is None:
            log.info('Min price for qkey [%s] was not found in YT', self.qkey)
            return None

        min_price = combined_data['min_price_variant_data'][0]
        value = min_price['national_price'] / 100.0

        currency_data = avia_data_currency_cache.by_id.get(
            min_price['national_currency_id']
        )

        if currency_data is None:
            log.warning('Unknown currency code %s', min_price['national_currency_id'])
            return None

        currency = currency_data.code

        log.info('Min price for qkey [%s] = %f %s', self.qkey, value, currency)
        return StoredMinPrice(
            value=value,
            currency=currency,
            stored_variants=combined_data,
        )


class MinPriceMysqlStorage(MinPriceStorage):
    def get_min_price(self, flight_type, passengers):
        log.info('Fetching min price from mysql')

        settlement_from = get_point_settlement(self.point_from)
        settlement_to = get_point_settlement(self.point_to)
        if not settlement_to or not settlement_from:
            return None

        search_kwargs = dict(
            national_version=self.national_version,
            departure_settlement_id=settlement_from.id,
            arrival_settlement_id=settlement_to.id,
            date_forward=self.departure_date,
            date_backward=self.return_date,
            passengers=passengers.format_key(),
            direct_flight=(flight_type == FlightType.Direct),
        )

        if self.currency_code is not None:
            search_kwargs['currency__code'] = self.currency_code

        min_prices = MinPrice.objects.filter(
            **search_kwargs
        ).order_by('price')[:1]

        if not min_prices:
            log.info('Min price is not found in mysql')
            return None

        [min_price] = min_prices
        log.info('Min price has been found in mysql: %r', min_price)

        return StoredMinPrice(
            value=min_price.price,
            currency=min_price.currency.code,
            stored_variants=StoredVariants.from_routes(min_price.routes).variants,
        )


# актуальный ключ минцен
# Как хранится в memcache? Json
def key_by_params2(
    city_from_key,
    city_to_key,
    national_version,
    passengers_key,
    date_forward,
    date_backward,
    is_direct,
):
    # Todo: копипаста из текущего демона, можно попробовать обновить сабмодуль,
    # но пока это страшно. Заменить на новый ключ стоит, чтоб не писать
    # из демона оба ключа
    return 'min_price_json_%s_%s_%s_%s_%s_%s_%s' % (
        city_from_key,
        city_to_key,
        national_version,
        passengers_key,
        date_forward.strftime('%Y-%m-%d'),
        date_backward and date_backward.strftime('%Y-%m-%d') or None,
        is_direct and 'direct' or 'transfers',
    )


class MinPriceMemcachedStorage(MinPriceStorage):
    def get_min_price(self, flight_type, passengers):
        log.info('Fetching min price from memcached')

        settlement_from = get_point_settlement(self.point_from)
        settlement_to = get_point_settlement(self.point_to)
        if not settlement_to or not settlement_from:
            return None

        key = key_by_params2(
            city_from_key=settlement_from.point_key,
            city_to_key=settlement_to.point_key,
            national_version=self.national_version,
            passengers_key=passengers.format_key(),
            date_forward=self.departure_date,
            date_backward=self.return_date,
            is_direct=(flight_type == FlightType.Direct),
        )
        cached = get_min_price_by_key(key)
        if not cached:
            log.info('Memcached miss [%s]', key)
            return None

        cached = ujson.loads(cached)
        log.info('Memcached hit [%s]: %r', key, cached)

        if self.currency_code is not None:
            if self.currency_code != cached['currency']:
                log.info(
                    'Currencies do not match: having %r while %r is needed',
                    cached['currency'], self.currency_code
                )
                return None

        routes = cached.get('routes', '')

        return StoredMinPrice(
            value=cached['price'],
            currency=cached['currency'],
            stored_variants=StoredVariants.from_routes(routes).variants,
        )


class MinPriceStorageChain(BaseMinPriceStorage):
    def __init__(self, *storages):
        self._storages = storages

    def warm_up(self, passengers):
        for storage in self._storages:
            storage.warm_up(passengers)

    def get_min_price(self, flight_type, passengers):
        for storage in self._storages:
            min_price = storage.get_min_price(flight_type, passengers)
            if min_price is not None:
                return min_price

        return None
