# -*- coding: utf-8 -*-
from __future__ import unicode_literals

import logging
import random
import time
from operator import attrgetter

from django.conf import settings

from travel.avia.library.python.common.geotargeting.lookup import get_city_by_ip
from travel.avia.library.python.common.models.geo import Settlement, Station
from travel.avia.library.python.common.models.partner import Partner, DohopVendor
from travel.avia.library.python.common.utils import marketstat

from travel.avia.avia_api.avia.cache.partners import partner_cache
from travel.avia.avia_api.avia.lib.decorators import skip_None_values

log = logging.getLogger(__name__)

show_log = marketstat.Log(settings.SHOW_LOG)
doubling_show_log = marketstat.DsvLog(settings.DOUBLING_SHOW_LOG)

popular_flights_log = marketstat.DsvLog(settings.YT_REDIR_LOG_PATH)


def write_show_log(
    variants, national_version, point_from, point_to, user_ip, platform,
    yandexuid
):
    sid_base = '%d.%d' % (
        random.randint(1000000, 9999999),
        time.time()
    )

    def _geoid(point):
        if isinstance(point, Settlement):
            return point._geo_id
        elif isinstance(point, Station):
            return point.settlement_id and point.settlement._geo_id or ''
        return None

    for index, variant in enumerate(sorted(variants, key=attrgetter('tariff'))):
        variant.show_id = '%s.%d' % (sid_base, index)

        extra = {
            'datasource_id': variant.partner.billing_datasource_id,
            'price': str(
                variant
                    .partner
                    .get_national_price(national_version)
                    .get_cents_dispersed()
            ),
            'url': getattr(variant, 'url', None),
            'city_from': _geoid(point_from),
            'city_to': _geoid(point_to),
            'host': 'api_avia',
            'show_id': variant.show_id,
            'national': national_version,
            'national_version': national_version,
            'pp': platform,
            'yandexuid': yandexuid,
        }

        show_log.write_event(
            userip=user_ip,
            extra=skip_None_values(extra)
        )

        doubling_extra = dict(
            extra,
            tskv_format='avia-show-log',
            **_billing_ids(variant.partner)
        )

        doubling_show_log.write_event(
            userip=user_ip,
            extra=doubling_extra
        )


def log_redirect(
    chance, partner, redirect_url, platform, user_ip, passportuid, yandexuid
):
    _write_popular_flights_log(chance, user_ip, passportuid, yandexuid)


def _write_popular_flights_log(chance, user_ip, passportuid, yandexuid):
    popular_flights_log.write_event(
        userip=user_ip,
        extra={
            'tskv_format': 'rasp-popular-flights-log',
            'service': 'api',

            'passportuid': passportuid,
            'yandexuid': yandexuid,
            'national_version': chance.national_version,
            'user_geoid': get_city_by_ip(user_ip)[1],

            'partner': chance.order_data['partner'],
            'tariff': _formatted_tariff_value(chance),

            'from_id': chance.from_point_key,
            'to_id': chance.to_point_key,
            'date_forward': chance.date_forward.strftime('%Y-%m-%d'),
            'date_backward': (
                chance.date_backward.strftime('%Y-%m-%d')
                if chance.date_backward is not None
                else None
            ),
            'forward_numbers': chance.forward_numbers,
            'backward_numbers': chance.backward_numbers,
        }
    )


def _formatted_tariff_value(chance):
    tariff = chance.order_data.get('tariff')
    if tariff is not None:
        value = tariff.get('value')
        currency = tariff.get('currency')
        if value is not None and currency is not None:
            return '{0:.2f} {1}'.format(value, currency)

    log.error('No tariff for chance %s', chance.encrypt())
    return None


def _billing_ids(partner):
    if isinstance(partner, DohopVendor):
        partner = partner_cache.by_code('dohop')

    if isinstance(partner, Partner):
        return {
            'billing_order_id': partner.billing_order_id,
            'billing_client_id': partner.billing_client_id,
        }

    log.critical('Unknown partner type: %r (%r)', type(partner), partner)

    return {
        'billing_order_id': getattr(partner, 'billing_order_id', None),
        'billing_client_id': getattr(partner, 'billing_client_id', None),
    }
