# -*- coding: utf-8 -*-
from __future__ import unicode_literals

import logging
from operator import itemgetter

from travel.library.python.tvm_ticket_provider import provider_fabric

from django.conf import settings


log = logging.getLogger(__name__)

# https://github.yandex-team.ru/common-python/python-blackbox
# http://doc.yandex-team.ru/blackbox/reference/method-oauth.xml

try:
    from blackbox import XmlBlackbox, BlackboxError
    blackbox = XmlBlackbox(url=settings.BLACKBOX_URL,
                           timeout=settings.BLACKBOX_TIMEOUT)

except Exception:
    log.critical('Blackbox error', exc_info=True)
    blackbox = None

tvm_client = provider_fabric.create(
    fake=not settings.ENABLE_TVM,
    source_id='avia-api',
    destinations=['blackbox'],
    renew=True,
)


def _get_tvm_header():
    ticket = tvm_client.get_ticket('blackbox')
    result = {}
    if ticket:
        result['X-Ya-Service-Ticket'] = ticket
    return result


def get_yandex_uid_by_oauth_token(oauth_token, userip):
    if not blackbox:
        raise BlackboxError('No blackbox')

    try:
        data = blackbox.oauth(
            oauth_token,
            userip=userip,
            by_token=True,
            headers=_get_tvm_header(),
        )
    except BlackboxError as exc:
        log.error('Blackbox: %s', exc)
        raise

    if data.get('valid', False):
        return data['uid']

    return None


def _request_validated_emails(uid):
    result = blackbox.userinfo(
        uid_or_login=uid,
        userip='127.0.0.1',
        emails='getall',
        headers=_get_tvm_header(),
    )
    if result is None or result['emails'] is None:
        return []
    else:
        return filter(itemgetter('validated'), result['emails'])


def get_emails_by_uid(uid):
    result = _request_validated_emails(uid)
    return map(itemgetter('address'), result)


def get_email_by_uid(uid):
    result = _request_validated_emails(uid)
    return next((x['address'] for x in result if x['default']), None)
