# -*- coding: utf-8 -*-
import logging
from collections import defaultdict

from django.conf import settings

from travel.avia.library.python.common.xgettext.i18n import tgettext

from travel.avia.avia_api.avia.lib.date import get_utc_now
from travel.avia.avia_api.avia.v1.model.constants import (
    AEROEXPRESS_EVENT, ARRIVAL_DATETIME, FLIGHT_KEY, DEPARTURE_DATETIME,
    TIME_CHANGED, FIELD, VALUE, GATE_ASSIGNED, FLIGHT_CANCELED, EVENT,
    CHECKIN_STARTED, PHONE_KEY, URL_KEY, MINIMAL_PRICE, FAVORITE_KEY, PRICE
)
from travel.avia.avia_api.avia.v1.schemas import ApiSchema, fields
from travel.avia.avia_api.avia.lib.pusher import Pusher
from travel.avia.avia_api.avia.lib.pusher.tag import PusherTag

log = logging.getLogger(__name__)


def get_users_by_flight(flight):
    from travel.avia.avia_api.avia.v1.model.user import User

    if flight.id is None:  # there are no users with not saved flights
        log.info(
            'Tried to generate notifications for a not saved flight %r', flight
        )
        return []

    return list(User.objects(flights=flight))


class PushNotification(object):
    default_ttl = 60
    tag = PusherTag.Undefined

    def get_message(self, lang):
        """
        :param str lang:
        :rtype: str
        """
        return None

    def get_data(self):
        return None

    def send(self, users, ttl=None):
        try:
            self._try_send(users, ttl)
        except Exception as exc:
            log.exception(
                "Couldn't send a push notification to %r with ttl %r: %r",
                users, ttl, exc
            )

    def _try_send(self, users, ttl=None):
        if ttl is None:
            ttl = self.default_ttl

        if not users:
            return

        data = {k: v for k, v in self.get_data().iteritems() if v}

        # Хак для юзеров, у которых ещё не завелась модель Device.
        # Подержать пару недель и удалить.
        empty_users = []
        hack_uuids = set()

        devices_by_languages = defaultdict(list)
        for user in users:
            if not user.devices:
                empty_users.append(user)
                hack_uuids.add(user.uuid)

            for device in user.devices:
                if device.uuid not in hack_uuids:
                    push_language = settings.TRANSLATED_LANGUAGES_MAPPING.get(
                        device.lang, 'ru'
                    )
                    devices_by_languages[push_language].append(device)

        # продолжение хака
        if 'ru' not in devices_by_languages:
            devices_by_languages['ru'] = []

        for lang, devices in devices_by_languages.items():
            message = self.get_message(lang)

            if message is None and data is None:
                log.error(
                    u'Got None data and [%r] message for [%r] lang and '
                    u'notification [%r]',
                    message, lang, self.__dict__
                )
                continue

            # конец хака
            users = empty_users if lang == 'ru' else []

            Pusher().push_many(
                users=users,
                devices=devices,
                data={
                    'u': data,
                    'title': message,
                },
                message=message,
                ttl=ttl,
                push_tag=self.tag,
            )


class MinPricePushSchema(ApiSchema):
    v = fields.Float(attribute='value', required=True)
    c = fields.String(attribute='currency', required=True)


class MinPricePushNotification(PushNotification):
    tag = PusherTag.MinPriceChange
    default_ttl = 60 * 60 * 2

    def __init__(self, favorite):
        self.favorite = favorite

    def get_message(self, lang):
        return tgettext(
            u'<point_from/> - <point_to/> от <price/>',
            point_from=self.favorite.point_from.L_title(lang=lang),
            point_to=self.favorite.point_to.L_title(lang=lang),
            price=self.favorite.min_price,
            lang=lang,
        )

    def get_data(self):
        min_price = self.favorite.min_price

        return {
            EVENT: MINIMAL_PRICE,
            FAVORITE_KEY: self.favorite.key,
            PRICE: MinPricePushSchema(strict=True).dump(min_price).data
        }


class FlightPushNotification(PushNotification):
    default_ttl = 60 * 60 * 24  # seconds in day

    def __init__(self, flight):
        self.flight = flight

    def get_data(self):
        return {
            FLIGHT_KEY: str(self.flight.id)
        }

    def send(self, users=None, ttl=None):
        if users is None:
            users = get_users_by_flight(self.flight)

        if not users:
            return

        aware_departure_datetime = (
            self.flight.airport_from.localize(self.flight.any_departure_dt)
        )

        seconds_to_departure = (
            aware_departure_datetime - get_utc_now()
        ).total_seconds()

        if seconds_to_departure <= 0:
            log.warning(
                'Tried to notify about a flight which took off: '
                '%r, arguments: %r, %r',
                self.flight, users, ttl
            )
            return

        if ttl is None:
            ttl = min(seconds_to_departure, self.default_ttl)

        super(FlightPushNotification, self).send(users, ttl)


class DepartureTimeChangePushNotification(FlightPushNotification):
    tag = PusherTag.FlightDepartureTimeChange

    def get_message(self, lang):
        return tgettext(
            u'Время вылета рейса <route/> изменено',
            route=self.flight.number,
            lang=lang
        )

    def get_data(self):
        data = super(DepartureTimeChangePushNotification, self).get_data()
        data.update({
            EVENT: TIME_CHANGED,
            FIELD: DEPARTURE_DATETIME,
            VALUE: self.flight.real_departure.strftime('%Y-%m-%d %H:%M'),
        })
        return data


class GateChangePushNotification(FlightPushNotification):
    tag = PusherTag.FlightGateChange

    def get_message(self, lang):
        return tgettext(
            u'Выход на посадку <gate/>, рейс <route/>',
            gate=self.flight.gate,
            route=self.flight.number,
            lang=lang
        )

    def get_data(self):
        data = super(GateChangePushNotification, self).get_data()

        try:
            flight_gate = int(self.flight.gate)
        except Exception:
            log.exception(
                'Bad gate on flight %s: %r',
                self.flight, self.flight.gate
            )
            return None

        data.update({
            EVENT: GATE_ASSIGNED,
            FIELD: 'gate',
            VALUE: flight_gate,
        })

        return data


class FlightCancellationPushNotification(FlightPushNotification):
    tag = PusherTag.FlightCancellation

    def get_message(self, lang):
        return tgettext(
            u'Рейс <route/> отменён',
            route=self.flight.number,
            lang=lang
        )

    def get_data(self):
        data = super(FlightCancellationPushNotification, self).get_data()
        data.update({
            EVENT: FLIGHT_CANCELED,
        })
        return data


class CheckinStartPushNotification(FlightPushNotification):
    tag = PusherTag.CheckinStart

    def get_message(self, lang):
        return tgettext(
            u'Зарегистрируйтесь на рейс <route/>',
            route=self.flight.number,
            lang=lang
        )

    def get_data(self):
        data = super(CheckinStartPushNotification, self).get_data()

        data.update({
            EVENT: CHECKIN_STARTED,
            URL_KEY: self.flight.company.registration_url,
            PHONE_KEY: self.flight.company.registration_phone,
        })

        return data


class AeroexpressPushNotification(FlightPushNotification):
    tag = PusherTag.Aeroexpress

    def get_message(self, lang):
        return tgettext(
            u'В <hours-minutes/> отходит aэроэкспресс',
            hours_minutes=(
                self.flight.aeroexpress
                    .local_departure_datetime.strftime('%H:%M')
            ),
            lang=lang,
        )

    def get_data(self):
        data = super(AeroexpressPushNotification, self).get_data()

        data.update({
            EVENT: AEROEXPRESS_EVENT,
            DEPARTURE_DATETIME: (self.flight.aeroexpress.local_departure_datetime.strftime('%Y-%m-%d %H:%M')),
            ARRIVAL_DATETIME: (self.flight.aeroexpress.local_arrival_datetime.strftime('%Y-%m-%d %H:%M')),
        })

        return data
