# -*- coding: utf-8 -*-
import json
import logging

from django.conf import settings
from typing import List, Dict

from travel.avia.avia_api.avia.lib.pusher.tag import PusherTag
from travel.avia.avia_api.avia.lib.pusher.transport import PushTransport, transport_by_name
from travel.avia.avia_api.avia.lib.pusher.exception import PusherError

log = logging.getLogger(__name__)


class Pusher(object):
    # Транспорт обратной совместимости. Если device.transport пусто, берем отсюда
    backward_transport = 'NullTransport'  # type: str

    def __init__(self, default_transport=None):
        # type: (PushTransport) -> None
        self.default_transport = (transport_by_name('Xiva') if default_transport is None
                                  else default_transport)  # type: PushTransport
        self.platforms = settings.PUSHER_PLATFORMS  # type: List[str]
        self.transports = {}  # type: Dict[str, PushTransport]

    def transport(self, name=None):
        # type: (str) -> PushTransport

        if name is None:
            return self.default_transport

        if name not in self.transports:
            self.transports[name] = transport_by_name(name)

        return self.transports[name]

    def add(self, device):
        # type: (Device) -> None
        from travel.avia.avia_api.avia.v1.model.device import Device

        try:
            if device.platform not in self.platforms:
                raise PusherError('Unknown platform "%s"' % device.platform)

            log.info('Add user: %r', [
                device.transport,
                device.push_token,
                device.uuid,
                device.user.yandex_uid,
                device.platform
            ])

            self.transport(device.transport).add_user(
                device.push_token,
                device.user.yandex_uid,
                device.uuid,
                device.platform
            )

        except PusherError as e:
            raise e

        except Exception as exc:
            log.critical(
                "Couldn't add user: [%s, %s] to %s: %s",
                device.uuid,
                device.user.yandex_uid,
                device.transport,
                exc,
                exc_info=True
            )

    def delete_by_uuid(self, uuid):
        # type: (str) -> None
        from travel.avia.avia_api.avia.v1.model.device import Device

        device_count = 0
        deleted_uuids = []
        for device in Device.objects(uuid=uuid):
            device_count += 1
            try:
                self.transport(
                    device.transport if device.transport else self.backward_transport
                ).delete(device.user.yandex_uid, device.uuid)
                device.push_token = None
                device.save()
            except Exception as exc:
                log.critical(
                    "Couldn't delete push user [%r]: %r",
                    uuid, exc,
                    exc_info=True
                )
            else:
                deleted_uuids.append(uuid)

        log.info(
            'Deleted %d/%d accounts with uuid=%r. Deleted: %r',
            len(deleted_uuids), device_count, uuid, deleted_uuids
        )

    def push_many(self, users=None, devices=None, data=None, message=None, ttl=60, push_tag=PusherTag.Undefined):
        # type: (List[User], List[Device], dict, str, int, str) -> None
        from travel.avia.avia_api.avia.v1.model.user import User
        from travel.avia.avia_api.avia.v1.model.device import Device

        users = users or []
        devices = devices or []

        # RASPTICKETS-3869
        # Добавить ко всем пушам магическую переменную
        data['pw_msg'] = 1
        if data.get('u'):
            data['u'] = json.dumps(data['u'], separators=(',', ':'))

        devices = set(devices or [])
        for u in users:
            devices |= set(u.devices)

        transports = {}
        for device in devices:
            # Если транспорт не указан, значит девайс "до ксивы", отправляем в топку
            transports.setdefault(
                device.transport if device.transport else self.backward_transport,
                set()
            ).add(device.uuid)

        for transport_name in transports:
            uuids = transports[transport_name]
            log.info('Push, uuids: %r, message: %r, data: %r',
                     uuids, message, data)

            push_sent = self.transport(transport_name).push(
                list(uuids), data, message=message, ttl=int(ttl), push_tag=push_tag
            )

            log.info(
                'Push stats: transport:%s, tag:%s, uuids:%d, pushes_sent:%d',
                transport_name, push_tag, len(uuids), push_sent
            )
