# -*- coding: utf-8 -*-
import logging
from abc import ABCMeta, abstractmethod
from typing import List

from django.utils.functional import cached_property
from six import add_metaclass

from travel.avia.avia_api.avia.lib.pusher.tag import PusherTag
from travel.avia.avia_api.avia.lib.pusher.exception import PusherError


@add_metaclass(ABCMeta)
class PushTransport(object):
    """Базовый класс транспорта пушей."""

    @abstractmethod
    def add_user(self, push_token, uid, uuid, platform):
        # type: (str, str, str, str) -> bool
        pass

    @abstractmethod
    def delete(self, uid, uuid):
        # type: (str) -> bool
        pass

    @abstractmethod
    def push(self, uuids, data, message=None, ttl=60, push_tag=PusherTag.Undefined):
        # type: (List[str], dict, str, int, str) -> int
        pass

    @cached_property
    def name(self):
        return self.__class__.__name__


class NullTransport(PushTransport):
    """Фейковый транспорт."""

    log = logging.getLogger(__name__)

    def add_user(self, push_token, uid, uuid, platform):
        self.log.info('Null transport add user: %r' % locals())
        return True

    def delete(self, uid, uuid):
        self.log.info('Null transport delete: %r' % locals())
        return True

    def push(self, uuids, data, message=None, ttl=60, push_tag=PusherTag.Undefined):
        self.log.info('Null transport push: %r' % locals())
        return 0


def transport_by_name(name):
    # type: (str) -> PushTransport

    if name == 'Xiva':
        from travel.avia.avia_api.avia.lib.pusher.transport.xiva import Xiva
        return Xiva()

    elif name == 'NullTransport':
        return NullTransport()

    else:
        raise PusherError('Unknown transport "%s"' % name)
