# -*- coding: utf-8 -*-
import requests
import logging
from functools import wraps
from typing import Dict, List

from django.conf import settings

from travel.avia.avia_api.avia.lib.pusher.transport import PushTransport
from travel.avia.avia_api.avia.lib.pusher.tag import PusherTag
from travel.avia.avia_api.avia.lib.pusher.exception import PusherError

log = logging.getLogger(__name__)


def check_xiva_response(method):
    @wraps(method)
    def decorator(*args, **kwargs):
        try:
            response = method(*args, **kwargs)  # type: requests.Response
        except XivaError as e:
            raise e
        except Exception as exc:
            log.critical('Xiva %s exception: %r', method.__name__, exc)
        else:
            if response.status_code == 200:
                log.info(
                    'Request %r %r. Answer: %r',
                    response.request.url,
                    response.request.body,
                    response.text
                )
            else:
                log.critical(
                    'Code: %s, content: %s, request: url=%r body=%r',
                    response.status_code,
                    response.text,
                    response.request.url,
                    response.request.body
                )
            return response
    return decorator


class XivaError(PusherError):
    pass


class Xiva(PushTransport):
    timeout = 5  # type: int

    def __init__(self, push_class=None):
        # type: (str) -> None

        self.app_name = settings.XIVA_APP_NAME  # type: Dict[str, str]
        self.host = settings.XIVA_HOST  # type: str
        self.subscribe_token = settings.XIVA_SUBSCRIBE_TOKEN  # type: str
        self.send_token = settings.XIVA_SEND_TOKEN  # type: str
        self.service = settings.XIVA_SERVICE  # type: str
        self.device_types = settings.XIVA_DEVICE_TYPES  # type: Dict[str, str]

        self.push_class = push_class if push_class is not None else XivaDefaultPush

    def add_user(self, push_token, uid, uuid, platform):
        self._register_device(uid, uuid, platform, push_token)

    def delete(self, uid, uuid):
        self._unregister_device(uid, uuid)

    def push(self, uuids, data, message=None, ttl=60, push_tag=PusherTag.Undefined):
        response = self._send_push(
            uuids,
            self.push_class(message, data),
            ttl=ttl,
            push_tag=push_tag
        )

        push_sent = 0
        for row in response.json().get('results', []):
            if row.get('code') == 200:
                push_sent += 1
                log.info('Push sent: %r' % row)
            else:
                log.warning('Push error: %r' % row)

        return push_sent

    def _request(self, method, params=None, data=None, json=None, headers=None):
        # type: (str, Dict, Dict, Dict, Dict) -> requests.Response

        return requests.post(
            'https://%s/v2%s' % (self.host, method),
            params=params,
            data=data,
            json=json,
            headers=headers,
            verify=False,
        )

    @check_xiva_response
    def _register_device(self, uid, uuid, platform, push_token):
        # type: (str, str, str, str) -> requests.Response

        if platform not in self.device_types.keys():
            raise XivaError('Unknown platform "%s"' % platform)

        device_type = self.device_types[platform]
        app_name = (self.app_name[device_type]
                    if device_type in self.app_name else self.app_name['default'])

        params = {
            'token': self.subscribe_token,
            'service': self.service,
            'user': uid if uid is not None else uuid,
            'uuid': uuid,
            'platform': device_type,
            'app_name': app_name,
        }

        data = {
            'push_token': push_token,
        }

        return self._request('/subscribe/app', params, data)

    @check_xiva_response
    def _unregister_device(self, uid, uuid):
        # type: (str) -> requests.Response

        params = {
            'token': self.subscribe_token,
            'service': self.service,
            'uuid': uuid,
            'user': uid if uid else uuid,
        }

        return self._request('/unsubscribe/app', params)

    @check_xiva_response
    def _send_push(self, uuids, push, ttl, push_tag):
        # type: (List[str], XivaDefaultPush, int, str) -> requests.Response

        params = {
            'token': self.send_token,
            'ttl': ttl,
            'event': push_tag,
        }

        body = {
            'recipients': uuids if type(uuids) is list else [uuids],
            'payload': push.get_payload(),
            'repack': push.get_repack(),
        }

        return self._request('/batch_send', params=params, json=body, headers={
            'X-DeliveryMode': 'queued',
        })


class XivaDefaultPush(object):
    def __init__(self, message, data=None):
        # type: (str, Dict) -> None

        self.message = message
        self.payload = data.get('payload', data) if isinstance(data, dict) else None
        self.repack = data.get('repack', {}) if isinstance(data, dict) else {}

    def get_payload(self):
        # type: () -> Dict

        return self.payload

    def get_repack(self):
        # type: () -> Dict

        if self.repack:
            return self.repack

        apns = {
            'aps': {
                'alert': self.message,
            },
        }

        if self.payload:
            apns['aps']['content-available'] = 1
            apns['repack_payload'] = self.payload.keys()

        return {
            'apns': apns,
            'gcm': {
                'title': self.message,
                'sound': 'default',
            },
            'wns': {
                'toast': self.message,
            },
            'mpns': {
                'toast': self.message,
            },
        }
