# -*- coding: utf-8 -*-
import base64
import logging
import zlib
from collections import OrderedDict
from datetime import datetime

import ujson
from django.db.models import Model

from travel.avia.library.python.common.utils.date import parse_date, get_pytz
from travel.avia.library.python.common.utils.exceptions import SimpleUnicodeException

from travel.avia.avia_api.avia.lib.helpers import get_serializer, DictWrapper

log = logging.getLogger(__name__)
DATETIME_FORMAT = '%Y-%m-%d %H:%M:%S'


class ToJsonFromJsonConverter(object):
    def __init__(self, to_json, from_json, name=None):
        self._to_json = to_json
        self._from_json = from_json
        self.name=name

    def to_json(self, obj, objects_cache=None):
        return self._to_json(obj)

    def from_json(self, obj, **kwargs):
        return self._from_json(obj)

    def __unicode__(self):
        return u'<%s>' % (self.name or self.__class__.__name__)


str_converter = ToJsonFromJsonConverter(str, str, name='str')

asis_converter = ToJsonFromJsonConverter(
    # не надо тут вызывать dumps&load
    # нужно просто возвращать объекты, но это сломает совместимость (
    ujson.dumps, ujson.loads,
    name='asis'
)

date_converter = ToJsonFromJsonConverter(
    lambda d: d and d.isoformat(),
    parse_date,
    name='date'
)

datetime_converter = ToJsonFromJsonConverter(
    lambda dt: dt and dt.strftime(DATETIME_FORMAT),
    lambda txt: txt and datetime.strptime(txt, DATETIME_FORMAT),
    name='datetime'
)

datetimeaware_converter = ToJsonFromJsonConverter(
    lambda dt: dt and {
        'local_datetime': dt.strftime(DATETIME_FORMAT),
        'tzname': dt.tzinfo.zone,
    },
    lambda dta: dta and get_pytz(dta['tzname']).localize(
        datetime.strptime(dta['local_datetime'], DATETIME_FORMAT)
    ),
    name='datetimeaware'
)


class Serializable(object):
    def dumps(self):
        raise NotImplementedError

    @classmethod
    def loads(cls, serialized):
        raise NotImplementedError


class Cryptable(Serializable):
    def encrypt(self):
        return get_serializer().dumps(self.dumps())

    @classmethod
    def decrypt(cls, crypted):
        return cls.loads(get_serializer().loads(crypted))

    @classmethod
    def create_from_crypted(cls, crypted, **kwargs):
        decrypted = cls.decrypt(crypted, **kwargs)

        log.debug('Cryptable.create_from_crypted decrypted: %s',
                  repr(decrypted))

        return cls.create_from_json(decrypted)

    def __str__(self):
        return self.encrypt()


class JsonSerializable(object):
    '''
    Базовый класс для сериализации и восстановления
    '''

    def __json__(self, objects_cache=None):
        return self.get_converter().to_json(self, objects_cache=objects_cache)

    @classmethod
    def create_from_json(cls, json_data, models_fetcher=None):
        if not models_fetcher:
            with ModelsFetcher() as models_fetcher:
                return cls.create_from_json(json_data, models_fetcher)

        return cls.get_converter().from_json(json_data,
                                             models_fetcher=models_fetcher)

    @classmethod
    def create_from_serialized(cls, serialized, *args, **kwargs):
        json_data = cls.loads(serialized)

        return cls.create_from_json(json_data, *args, **kwargs)

    @classmethod
    def get_converter(cls):
        return ClassConverter(cls)

    def dumps(self, *args, **kwargs):
        objects_cache = ObjectsCache()
        return ujson.dumps(self.__json__(objects_cache=objects_cache))

    @classmethod
    def loads(cls, val):
        return ujson.loads(val)


class JsonCryptable(JsonSerializable, Cryptable):
    pass


class CsvSerializable(JsonSerializable):
    delimiter = ','

    @classmethod
    def get_converter(cls):
        if not hasattr(cls, '_converter'):
            cls._converter = SimpleClassConverter(cls)

        return cls._converter

    def dumps(self, *args, **kwargs):
        json_data = self.__json__()
        what_attrs = self.get_converter().what_attrs()
        values = [json_data.get(k) for k in what_attrs]

        return self.delimiter.join(values)

    @classmethod
    def loads(cls, val):
        values = val.split(cls.delimiter)
        what_attrs = cls.get_converter().what_attrs()

        result = type(what_attrs)(zip(what_attrs.keys(), values))

        log.debug('CsvSerializable.loads result: %s', repr(result))

        return result


class CsvCryptable(CsvSerializable, Cryptable):
    pass


class JsonPackable(JsonSerializable):
    def pack(self):
        return zlib.compress(self.dumps())

    @classmethod
    def unpack(cls, packed):
        ''' return json representation of cls '''

        return cls.loads(zlib.decompress(packed))

    @classmethod
    def create_from_packed(cls, packed, **kwargs):
        return cls.create_from_json(cls.unpack(packed), **kwargs)

    def pack_base64(self):
        return base64.b64encode(self.pack())

    @classmethod
    def create_from_base64(cls, base64_encoded, **kwargs):
        packed = base64.b64decode(base64_encoded)

        return cls.create_from_packed(packed, **kwargs)


class JsonAttrsConverter(object):
    def __init__(self, attrs):
        self.__attrs = attrs

    def what_attrs(self, obj=None, target_cls=None):
        return self.__attrs

    def to_json(self, obj, objects_cache=None):
        if obj is None:
            return json_None

        json_data = {}

        what_attrs = self.what_attrs(obj=obj)

        for key, converter in what_attrs.items():
            try:
                json_value = converter.to_json(getattr(obj, key, None), objects_cache=objects_cache)

            except ConvertToJsonError:
                # Чтобы не логировались обратно по рекурсии
                raise

            except Exception as e:
                msg = u'attrs_to_json. key: [%s], converter: %s. %s' % \
                    (key, converter, repr(e))

                log.exception(msg)

                raise ConvertToJsonError(msg)

            json_data[key] = json_value

        return json_data

    def from_json(self, json_data, models_fetcher):
        if json_data == json_None:
            return None

        target_cls = self.get_target_cls(json_data)
        what_attrs = self.what_attrs(target_cls=target_cls)
        attrs = self.attrs_from_json(json_data, what_attrs, models_fetcher)

        return construct_cls_from_attrs(target_cls, attrs, models_fetcher)

    def get_target_cls(self, json_data):
        return DictWrapper

    def attrs_from_json(self, json_data, what_attrs, models_fetcher):
        attrs = {}

        for key, converter in what_attrs.items():
            if key not in json_data:
                continue

            json_value = json_data[key]

            try:
                value = converter.from_json(json_value, models_fetcher=models_fetcher)

            except ConvertFromJsonError:
                # Чтобы не логировались обратно по рекурсии
                raise

            except Exception as e:
                msg = (u'attrs_from_json. obj: %s, key: [%s], converter: %s. %s' %
                       (self, key, converter, repr(e)))

                log.exception(msg)

                raise ConvertFromJsonError(msg)

            attrs[key] = value

        return attrs

    def __unicode__(self):
        return u'<%s: %s>' % (self.__class__.__name__, self.__attrs.keys())


class SimpleClassConverter(JsonAttrsConverter):
    def __init__(self, cls):
        if not hasattr(cls, '_serialize_attrs'):
            raise SerializationError(
                u'Класс %s должен иметь _serialize_attrs '
                u'для использования в %s' % (
                    cls.__name__, self.__class__.__name__
                )
            )

        self.cls = cls

    def what_attrs(self, *args, **kwargs):
        if not hasattr(self, '_what_attrs'):
            self._what_attrs = OrderedDict(self.cls._serialize_attrs)

        return self._what_attrs

    def get_target_cls(self, *args, **kwargs):
        return self.cls

    def __unicode__(self):
        return u'<%s: %s>' % (self.__class__.__name__, self.cls)


class ClassConverter(JsonAttrsConverter):
    def __init__(self, cls):
        if not hasattr(cls, '_json_attrs'):
            raise SerializationError(
                u'Класс %s должен иметь _json_attrs для использования в %s' % (
                    cls.__name__, self.__class__.__name__
                )
            )

        self.cls = cls

        if hasattr(self.cls, '_get_subclasses'):
            self.maintained_classes = self.cls._get_subclasses()

        else:
            self.maintained_classes = [self.cls]

    def what_attrs(self, **kwargs):
        if 'obj' in kwargs:
            obj = kwargs['obj']

            source_cls = obj.__class__

            if source_cls not in self.maintained_classes:
                raise NotAlowedToConvertClass(source_cls)

            return source_cls._json_attrs

        elif 'target_cls' in kwargs:
            return kwargs['target_cls']._json_attrs

        raise SerializationError(u'what_attrs нужно вызывать с obj '
                                 u'или target_cls keyword-аргументами')

    def get_target_cls(self, json_data):
        try:
            cls_name = json_data['__class_name__']

        except KeyError:
            if len(self.maintained_classes) == 1:
                cls_name = self.maintained_classes[0].__name__

            else:
                raise MissedClassName(json_data)

        cls_choices = {c.__name__: c for c in self.maintained_classes}

        try:
            return cls_choices[cls_name]

        except KeyError:
            raise NotAlowedToRestoreClass(cls_name)

    def to_json(self, obj, objects_cache=None):
        def base_to_json():
            json_data = super(ClassConverter, self).to_json(obj, objects_cache)

            if isinstance(json_data, dict):
                json_data['__class_name__'] = obj.__class__.__name__

            return json_data

        if objects_cache and hasattr(obj, 'cache_key'):
            key = obj.cache_key()

            if key:
                return objects_cache.get_object(self.cls, key, base_to_json)

        return base_to_json()

    def from_json(self, json_data, models_fetcher):
        def base_from_json():
            return super(ClassConverter, self).from_json(json_data, models_fetcher)

        if hasattr(self, 'cls') and hasattr(self.cls, '_cache_key_func'):
            key = self.cls._cache_key_func(json_data)

            if key:
                return models_fetcher.get_object(self.cls, key, base_from_json)

        return base_from_json()

    def __unicode__(self):
        return u'<%s: %s>' % (self.__class__.__name__, self.cls)


class ModelConverter(ClassConverter):
    def __init__(self, maintained, attrs=None):
        if isinstance(maintained, (list, tuple)):
            self.maintained_classes = maintained

        else:
            self.maintained_classes = [maintained]

        self.__attrs = attrs or {}

        # Можем положить так, значение primary_key первой попавшейся модели
        # потому что практически всегда оно будет==id, а отличаться будет
        # только в особых случаях наподобие AviaCompany, которую мы сериализуем
        # в одиночку.
        pk_field = self.maintained_classes[0]._meta.pk.attname

        if pk_field not in self.__attrs:
            self.__attrs[pk_field] = asis_converter

    def what_attrs(self, obj=None, target_cls=None):
        return self.__attrs

    def __unicode__(self):
        return u'<%s: %s>' % (self.__class__.__name__, self.maintained_classes)


class IfExistsConverter(object):
    def __init__(self, itemconverter):
        self.itemconverter = itemconverter

    def to_json(self, item, objects_cache=None):
        if item is None:
            return None

        return self.itemconverter.to_json(item, objects_cache=objects_cache)

    def from_json(self, item, **kwargs):
        if item is None:
            return None

        if item == json_None:
            return None

        return self.itemconverter.from_json(item, **kwargs)

    def __unicode__(self):
        return u'<%s itemconverter=%s>' % (self.__class__.__name__,
                                           self.itemconverter)


class ListConverter(object):
    def __init__(self, itemconverter):
        self.itemconverter = itemconverter

    def to_json(self, items, objects_cache=None):
        return [self.itemconverter.to_json(item, objects_cache=objects_cache) for item in items]

    def from_json(self, items, **kwargs):
        return [
            self.itemconverter.from_json(item, **kwargs)
            for item in items
        ]

    def __unicode__(self):
        return u'<%s itemconverter=%s>' % (self.__class__.__name__,
                                           self.itemconverter)


class DictConverter(object):
    def __init__(self, keyconverter, itemconverter):
        self.keyconverter = keyconverter
        self.itemconverter = itemconverter

    def to_json(self, items, objects_cache=None):
        return [(self.keyconverter.to_json(key, objects_cache=objects_cache),
                 self.itemconverter.to_json(item, objects_cache=objects_cache))
                for key, item in items.items()]

    def from_json(self, items, **kwargs):
        return {
            self.keyconverter.from_json(key, **kwargs): self.itemconverter.from_json(item, **kwargs)
            for key, item in items
        }

    def __unicode__(self):
        return u'<%s keyconverter=%s itemconverter=%s>' % (
            self.__class__.__name__, self.keyconverter, self.itemconverter
        )


class ObjectsCache(object):
    def __init__(self):
        self.objects_cache = {}

    def get_object(self, cls, key, getter):
        if cls not in self.objects_cache:
            self.objects_cache[cls] = {}

        if key not in self.objects_cache[cls]:
            self.objects_cache[cls][key] = getter()

        return self.objects_cache[cls][key]


class ModelsFetcher(ObjectsCache):
    def __init__(self):
        super(ModelsFetcher, self).__init__()
        self.models_cache = {}

    def __enter__(self):
        return self

    def __exit__(self, exc_type, exc_val, exc_tb):
        if exc_type is None:
            self.fetch_constructed()

    def construct_from_attrs(self, model_cls, attrs):
        pkname = model_cls._meta.pk.attname
        pk_ = attrs.get(pkname, None)

        if not pk_:
            return None

        if model_cls not in self.models_cache:
            self.models_cache[model_cls] = {}

        if pk_ not in self.models_cache[model_cls]:
            self.models_cache[model_cls][pk_] = model_cls(**attrs)

        return self.models_cache[model_cls][pk_]

    def fetch_constructed(self):
        ''' Достать из базы аттрибуты созданных моделей '''

        for model_cls, billets in self.models_cache.items():
            bulk = model_cls.objects.in_bulk(billets.keys())

            for billet in billets.values():
                if billet.pk in bulk:
                    src = bulk[billet.pk]

                    for field in src._meta.fields:
                        attr = field.attname
                        setattr(billet, attr, getattr(src, attr))


json_None = ujson.dumps(None)


class NotAlowedToConvertClass(SimpleUnicodeException):
    pass


class NotAlowedToRestoreClass(SimpleUnicodeException):
    pass


class MissedClassName(SimpleUnicodeException):
    pass


class SerializationError(SimpleUnicodeException):
    pass


class ConvertToJsonError(SimpleUnicodeException):
    pass


class ConvertFromJsonError(SimpleUnicodeException):
    pass


def construct_cls_from_attrs(target_cls, attrs, models_fetcher=None):
    if issubclass(target_cls, Model):
        if not models_fetcher:
            raise Exception('ModelsFetcher instance must be provided '
                            'to construct model %s' % target_cls.__name__)

        obj = models_fetcher.construct_from_attrs(target_cls, attrs)

    elif issubclass(target_cls, dict):
        obj = target_cls(**attrs)

    else:
        obj = target_cls.__new__(target_cls)

        for key, val in attrs.items():
            try:
                setattr(obj, key, val)

            except ValueError as e:
                log.error(u'Setting %s instance attribute: %s', target_cls, e)

                raise

    return obj
