# -*- coding: utf-8 -*-
import logging
import pytz
from datetime import datetime, date
from typing import Optional

from django.conf import settings
from marshmallow import post_load

from travel.avia.library.python.shared_flights_client.client import SharedFlightsClient

from travel.avia.avia_api.avia.v1.schemas import fields, ApiSchema
from travel.avia.avia_api.avia.lib.flight_info import WizardFlightInfo, WizardFlightStatus
from travel.avia.avia_api.avia.cache.stations import station_cache
from travel.avia.avia_api.avia.cache.settlements import settlement_cache

log = logging.getLogger(__name__)


class SharedFlightsFlightStatusSchema(ApiSchema):
    STATUS_MAP = {
        'unknown': 'unknown',
        'cancelled': 'cancelled',
        'arrived': 'arrive',
        'delayed': 'late',
        'early': 'wait',
        'on_time': 'wait',
    }
    # TODO: Переделать на логику из лендинга рейса или колдунщика
    STATUS_TITLE_MAP = {
        'unknown': '',
        'cancelled': 'Отменен',
        'arrived': 'Прибыл',
        'delayed': 'Задерживается',
        'early': 'early',
        'on_time': 'on_time',
    }

    status = fields.String(required=True)
    arrival = fields.DateTimeBlank(required=False, format='%Y-%m-%d %H:%M:%S')
    departure = fields.DateTimeBlank(required=False, format='%Y-%m-%d %H:%M:%S')


class SharedFlightsFlightSchema(ApiSchema):
    airline_iata = fields.String(required=True, load_from='airlineCode')
    airline_id = fields.Integer(required=True, load_from='airlineID')
    number = fields.String(required=True)
    departure_day = fields.DateYMD(required=True, load_from='departureDay')
    departure_time = fields.Time(required=True, load_from='departureTime')
    departure_timezone = fields.String(required=True, load_from='departureTimezone')
    arrival_day = fields.DateYMD(required=True, load_from='arrivalDay')
    arrival_time = fields.Time(required=True, load_from='arrivalTime')
    arrival_timezone = fields.String(required=True, load_from='arrivalTimezone')
    airport_from_code = fields.String(required=True, load_from='airportFromCode')
    airport_from_id = fields.Integer(required=True, load_from='airportFromID')
    airport_to_code = fields.String(required=True, load_from='airportToCode')
    airport_to_id = fields.Integer(required=True, load_from='airportToID')
    status = fields.Nested(SharedFlightsFlightStatusSchema)

    @post_load
    def make_wizard_info(self, data):
        # type: (dict) -> WizardFlightInfo
        flight_status = data['status']
        airport_from = station_cache.by_id(data['airport_from_id'])
        settlement_from = settlement_cache.by_id(airport_from.settlement_id) if airport_from else None
        airport_to = station_cache.by_id(data['airport_to_id'])
        settlement_to = settlement_cache.by_id(airport_to.settlement_id) if airport_to else None

        scheduled_departure_datetime = pytz.timezone(data['departure_timezone']).localize(
            datetime.combine(data['departure_day'], data['departure_time'])
        )
        scheduled_arrival_datetime = pytz.timezone(data['arrival_timezone']).localize(
            datetime.combine(data['arrival_day'], data['arrival_time'])
        )

        try:
            real_departure_datetime = pytz.timezone(data['departure_timezone']).localize(
                flight_status['departure']
            )
        except (KeyError, AttributeError):
            real_departure_datetime = scheduled_departure_datetime

        try:
            status = WizardFlightStatus(SharedFlightsFlightStatusSchema.STATUS_MAP[flight_status['status']])
            status_title = SharedFlightsFlightStatusSchema.STATUS_TITLE_MAP[flight_status['status']]
        except KeyError:
            status = WizardFlightStatus(WizardFlightStatus.Nodata)
            status_title = ''

        return WizardFlightInfo(
            number=data['number'],
            scheduled_departure_datetime=scheduled_departure_datetime,
            scheduled_arrival_datetime=scheduled_arrival_datetime,
            real_departure_datetime=real_departure_datetime,
            company_id=data['airline_id'],
            status=status,
            status_title=status_title,
            from_station_id=data['airport_from_id'],
            from_iata=data['airport_from_code'],
            from_settlement_geoid=settlement_from.get_settlement_geo_id() if settlement_from else None,
            to_station_id=data['airport_to_id'],
            to_iata=data['airport_to_code'],
            to_settlement_geoid=settlement_to.get_settlement_geo_id() if settlement_to else None,
        )


class SharedFlights(object):
    _REQUEST_TIMEOUT = 3

    def __init__(self, api_url):
        # type: (str) -> None
        self._client = SharedFlightsClient(api_url)

    def get_flight(self, number, departure_date):
        # type: (str, date) -> Optional[WizardFlightInfo]
        try:
            if not number:
                return None

            flight_iata, flight_number = number.split(' ', 2)
            json_data = self._client.flight(flight_iata, flight_number, departure_date)
            if json_data is None:
                return None

            flight, errors = SharedFlightsFlightSchema().load(json_data)
            if errors:
                raise Exception('Bad response: %r %r', json_data, errors)

            return flight

        except Exception as exc:
            log.exception(
                u'Get flight info error. Number: %s, exception: %s',
                number,
                exc,
            )

        return None

    def get_flights_network(self, carriers):
        # type: (list[str]) -> dict
        return self._client.flights_network(carriers)

    def get_aeroflot_variants(self, depart_from, arrive_to, after, before, national_version, is_one_way):
        # type: (list[str], list[str], str, str, str, bool) -> dict
        return self._client.aeroflot_variants(depart_from, arrive_to, after, before, national_version, is_one_way)

shared_flights = SharedFlights(settings.SHARED_FLIGHTS_API_URL)
