# -*- coding: utf-8 -*-
import logging
from datetime import date

from travel.avia.avia_api.avia.v1.model.flight import Flight
from travel.avia.avia_api.avia.lib.shared_flights import shared_flights

log = logging.getLogger(__name__)


class UnknownFlightException(Exception):
    def __init__(self, number, departure_date):
        # type: (str, date) -> None
        super(UnknownFlightException, self).__init__(self)
        self.number = number
        self.departure_date = departure_date


def add_flight_manually(user, number, departure_date):
    flights = []

    wizard_info = shared_flights.get_flight(number, departure_date)
    if not wizard_info:
        log.info("Couldn't get any info for %r;%r", number, departure_date)
        raise UnknownFlightException(number, departure_date)

    try:
        flight = Flight.objects.get(
            number=number,
            departure_date=departure_date,
            departure=wizard_info.departure_station_key(),
        )
    except Flight.DoesNotExist:
        # RASPTICKETS-4923 - забираем из колдунщика только те рейсы,
        # которых ещё нет в базе данных.
        flight = Flight(
            number=number,
            departure_date=departure_date,
            departure=wizard_info.departure_station_key(),
        ).updated_with_info(
            wizard_info
        )

    flights.append(flight)

    if not flights:
        log.info("Couldn't get wizard info for %r;%r", number, departure_date)
        flights.extend(Flight.objects(
            number=number,
            departure_date=departure_date,
        ))

    if not flights:
        log.info("Couldn't get any info for %r;%r", number, departure_date)
        raise UnknownFlightException(number, departure_date)

    user.update(pull_all__flights=flights)
    user.update(add_to_set__flights=flights)

    for flight in flights:
        flight.update_aeroexpress()
        if flight.aeroexpress:
            user.update(pull__aeroex_notify_flights=flight)
            user.update(add_to_set__aeroex_notify_flights=flight)
            log.info(
                'User %r will be notified about the aeroexpress for %r',
                user, flight
            )
            flight.user_aeroexpress = flight.aeroexpress
            user.reload()

    return {
        'flights': flights,
    }
