# -*- coding: utf-8 -*-


class VariantsReference(object):
    def __init__(self, variants):
        self._variants = variants

    def variants(self):
        return tuple(self._variants)

    def trips(self):
        return tuple({
            trip.key(): trip
            for v in self._variants
            for trip in [v.forward, v.backward]
            if trip.segments
        }.itervalues())

    def segments(self):
        return tuple({
            (
                segment.number,
                segment.station_from_iata,
                segment.station_to_iata,
                segment.local_departure,
                segment.local_arrival,
            ): segment
            for trip in self.trips()
            for segment in trip.segments
        }.itervalues())

    def partners(self):
        return tuple({
            v.partner.code: v.partner
            for v in self._variants
        }.itervalues())

    def companies(self):
        return tuple({
            segment.company.id: segment.company
            for v in self._variants
            for trip in [v.forward, v.backward]
            for segment in trip.segments
        }.itervalues())


class FictionalCompany(object):
    def __init__(self, company_code):
        self._company_code = company_code
        self.id = self.magic_company_id(company_code)
        self.iata = company_code
        self.sirena_id = company_code
        self.logo = None
        self.svg_logo2 = None
        self.svg2png_logo2 = None
        self.iata_priority = 0

    def L_title(self, lang):
        return self._company_code

    @staticmethod
    def magic_company_id(code):
        code = code.strip().lower()
        utf8_code = code.encode('utf-8') if isinstance(code, unicode) else code
        return 3 * 10**8 + int(utf8_code.encode('hex'), 16)


class SegmentWithCompanyFallback(object):
    def __init__(self, segment):
        self._segment = segment
        self._company = segment.company or FictionalCompany(
            company_code=self._segment.number.split()[0]
        )

    def __getattr__(self, item):
        return getattr(self._segment, item)

    @property
    def company(self):
        return self._company


class VariantsReferenceWithCompanyFallback(object):
    def __init__(self, variants_reference):
        self._reference = variants_reference

    def segments(self):
        return tuple(
            SegmentWithCompanyFallback(segment)
            for segment in self._reference.segments()
        )

    def companies(self):
        return tuple({
            segment.company.id: segment.company
            for segment in self.segments()
        }.itervalues())

    def variants(self):
        return self._reference.variants()

    def trips(self):
        return self._reference.trips()

    def partners(self):
        return self._reference.partners()


class HttpVariantsReference(object):
    def __init__(self, variants):
        self._variants = variants

    def variants(self):
        return tuple(self._variants)

    def trips(self):
        return tuple(
            {
                trip.key(): trip
                for variant in self._variants
                for trip in (variant.forward, variant.backward)
                if trip
            }.itervalues()
        )

    def partners(self):
        return tuple({
            v.partner.code: v.partner
            for v in self._variants
        }.itervalues())

    def companies(self):
        return tuple({
            company.id: company
            for company in (
                flight.company
                for trip in self.trips()
                for flight in trip.segments
            )
            if company
        }.itervalues())

    def segments(self):
        return tuple({
            (
                segment.number,
                segment.departure_station,
                segment.arrival_station,
                segment.departs_at,
                segment.arrives_at,
            ): segment
            for trip in self.trips()
            for segment in trip.segments
        }.itervalues())


class HttpSegmentWithCompanyFallback(object):
    def __init__(self, segment):
        self._segment = segment
        self._company = segment.company or FictionalCompany(
            company_code=self._segment.number.split()[0]
        )

    def __getattr__(self, item):
        return getattr(self._segment, item)

    @property
    def company(self):
        return self._company


def inject_fictional_companies(http_variants):
    unique_trips = {
        trip
        for variant in http_variants
        for trip in (variant.forward, variant.backward)
    }
    for trip in unique_trips:
        trip.segments = [
            HttpSegmentWithCompanyFallback(flight)
            for flight in trip.segments
        ]
