# -*- coding: utf-8 -*-
import logging
import time
from enum import Enum

import ujson

from travel.avia.avia_api.avia.lib.yt_loggers.yt_log import validate_schema

user_action_yt_logger = logging.getLogger('yt.subscriptions.user_action')
log = logging.getLogger(__name__)


class UserActionType(Enum):
    SUBSCRIBE = 'subscribe'
    UNSUBSCRIBE = 'unsubscribe'
    UNSUBSCRIBE_DIRECTION = 'unsubscribe_direction'
    CONFIRM = 'confirm'


user_action_log_schema = dict(
    unixtime=dict(type=int, lf_type='VT_UINT64', required=True),
    action_type=dict(type=UserActionType, lf_type='VT_STRING', required=True),
    email=dict(type=basestring, lf_type='VT_STRING', required=True),
    qkey=dict(type=basestring, lf_type='VT_STRING', required=False),
    date_range=dict(type=int, lf_type='VT_INT64', required=False),
    filter_params=dict(type=dict, lf_type='VT_ANY', required=False),
    source=dict(type=basestring, lf_type='VT_STRING', required=False),
    pending_passport=dict(type=basestring, lf_type='VT_STRING', required=False),
    pending_session=dict(type=basestring, lf_type='VT_STRING', required=False),
    min_price=dict(type=dict, lf_type='VT_ANY', required=False),
)


def log_user_action(**kwargs):
    """
    :param kwargs: logging data according to user_action_log_schema
    """
    try:
        kwargs.update(dict(unixtime=int(time.time())))
        validate_schema(kwargs, user_action_log_schema)
        user_action_yt_logger.info('%s', ujson.dumps(kwargs))
    except (KeyError, ValueError, TypeError):
        log.exception('Cannot log data %s', kwargs)


def log_user_subscribed(
    email=None, qkey=None, date_range=None, filter_params=None, source=None,
    pending_passport=None, pending_session=None, min_price=None,
):
    """
    :param basestring email:
    :param basestring qkey:
    :param int date_range:
    :param dict filter_params:
    :param basestring source:
    :param basestring pending_passport:
    :param basestring pending_session:
    :param tuple[int|None, str|None] min_price:
    :return:
    """
    log_user_action(
        action_type=UserActionType.SUBSCRIBE,
        email=email,
        qkey=qkey,
        date_range=date_range,
        filter_params=filter_params,
        source=source,
        pending_passport=pending_passport,
        pending_session=pending_session,
        min_price=min_price,
    )


def log_user_confirm(
    email=None, qkey=None, date_range=None, filter_params=None,
    pending_passport=None, pending_session=None
):
    """
    :param basestring email:
    :param basestring qkey:
    :param int date_range:
    :param dict filter_params:
    :param basestring pending_passport:
    :param basestring pending_session:
    :return:
    """
    log_user_action(
        action_type=UserActionType.CONFIRM,
        email=email,
        qkey=qkey,
        date_range=date_range,
        filter_params=filter_params,
        pending_passport=pending_passport,
        pending_session=pending_session,
    )


def log_user_unsubscribe(
    email=None,
):
    """
    :param str email:
    :return:
    """
    log_user_action(
        action_type=UserActionType.UNSUBSCRIBE,
        email=email,
    )


def log_user_unsubscribe_direction(
    email=None, qkey=None, date_range=None, filter_params=None,
    pending_passport=None, pending_session=None
):
    """
    :param str email:
    :param str qkey:
    :param int date_range:
    :param dict filter_params:
    :param str pending_passport:
    :param str pending_session:
    :return:
    """
    log_user_action(
        action_type=UserActionType.UNSUBSCRIBE_DIRECTION,
        email=email,
        qkey=qkey,
        date_range=date_range,
        filter_params=filter_params,
        pending_passport=pending_passport,
        pending_session=pending_session,
    )
